import React from 'react';

import { UseFormControllerSub } from 'shared/hooks/useFormController/useFormController';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';

export type UseFormValidation<T, K extends keyof T = keyof T> = Partial<
    Record<K, (value: T[K], store: T) => Optional<FormErrorSchema>>
>;

export interface UseFormOptions<T, K extends keyof T = keyof T> {
    init: T;
    validation: UseFormValidation<T, K>;

    onFormChange?(state: T): void;

    controller?: UseFormControllerSub<T>;
}

export interface UseFormResult<T, K extends keyof T = keyof T> {
    getValues(): T;

    getValue<K extends keyof T>(key: K): T[K];

    setValues(data: Partial<T>): void;

    setValue<K extends keyof T>(key: K, value: T[K]): void;

    setError<K extends keyof T>(key: K, errorsList: Optional<FormErrorSchema>): void;

    update<K extends keyof T>(key: K): (value: T[K]) => void;

    watch<K extends keyof T>(key: K): T[K];

    validate(): boolean;

    isValid(): boolean;

    readonly errors: Partial<Record<K, FormErrorSchema>>;
}

export function useForm<T, K extends keyof T = keyof T>(options: UseFormOptions<T, K>): UseFormResult<T, K> {
    const [, setVer] = React.useState<number>(0);

    const store = React.useRef<T>({ ...options.init });
    const errors = React.useRef<Partial<Record<K, FormErrorSchema>>>({});
    const watchlist = React.useRef<Partial<Record<K, boolean>>>({});
    const updaters = React.useRef<Partial<Record<K, (value: T[K]) => void>>>({});

    const rerender = React.useCallback(() => {
        setVer((ver) => ver + 1);
    }, [setVer]);

    const processChange = React.useCallback(
        (key: K) => {
            let scheduleRender = false;

            if (errors.current[key]) {
                errors.current[key] = undefined;
                scheduleRender = true;
            }

            if (watchlist.current[key]) {
                scheduleRender = true;
            }

            return scheduleRender;
        },
        [errors, watchlist],
    ) as <K extends keyof T>(key: K) => boolean;

    const onChange = React.useCallback(
        (key: K, value: T[K]) => {
            const scheduleRender = processChange(key);

            if (scheduleRender) {
                rerender();
            }

            if (options.onFormChange) {
                options.onFormChange(store.current);
            }
        },
        [store, processChange, rerender, options.onFormChange],
    );

    // reset updaters cache
    React.useEffect(() => {
        updaters.current = {};
    }, [store, onChange]);

    // set bulk values
    const setValues = React.useCallback(
        (data: Partial<T>) => {
            store.current = { ...store.current, ...data };

            const keys = Object.keys(data) as Array<keyof T>;
            let scheduleRender = false;

            for (const key of keys) {
                if (processChange(key)) {
                    scheduleRender = true;
                }
            }

            if (scheduleRender) {
                rerender();
            }

            if (options.onFormChange) {
                options.onFormChange(store.current);
            }
        },
        [store, processChange, rerender, options.onFormChange],
    );

    // lazy created updater functions builder
    const update = React.useCallback(
        (key: K) => {
            if (!updaters.current[key]) {
                updaters.current[key] = function updater(value: T[K]) {
                    store.current = { ...store.current, [key]: value };
                    onChange(key, value);
                };
            }

            return updaters.current[key];
        },
        [updaters, store, onChange],
    ) as <K extends keyof T>(key: K) => (value: T[K]) => void;

    const getValues = React.useCallback(() => {
        return { ...store.current };
    }, [store]);

    const getValue = React.useCallback(
        (key: K) => {
            return store.current[key];
        },
        [store],
    ) as <K extends keyof T>(key: K) => T[K];

    const setValue = React.useCallback(
        (key: K, value: T[K]) => {
            update(key)(value);
        },
        [updaters],
    ) as <K extends keyof T>(key: K, value: T[K]) => void;

    const watch = React.useCallback(
        (key: K) => {
            watchlist.current[key] = true;

            return getValue(key);
        },
        [watchlist, getValue],
    ) as <K extends keyof T>(key: K) => T[K];

    const isValid = React.useCallback(() => {
        return !Object.keys(errors.current).some((key) => {
            const value = errors.current[key];

            return Boolean(value && value.length);
        });
    }, [errors]);

    const validate = React.useCallback(() => {
        errors.current = Object.keys(options.validation).reduce((memo, key) => {
            memo[key] = options.validation[key](store.current[key], store.current);

            return memo;
        }, {});

        rerender();

        return isValid();
    }, [store, options.validation, errors, isValid, rerender]);

    const setError = React.useCallback(
        (key, errorsList) => {
            // @ts-ignore
            errors.current[key] = errorsList;

            rerender();
        },
        [errors],
    ) as <K extends keyof T>(key: K, value: Optional<FormErrorSchema>) => void;

    React.useEffect(() => {
        const controller = options.controller;
        let off: Array<() => void> = [];

        if (controller) {
            off.push(controller.on('getValues', getValues));
            off.push(controller.on('validate', validate));
            off.push(controller.on('isValid', isValid));
            off.push(controller.on('setError', setError));
        }

        return () => {
            off.forEach((item) => item());
        };
    }, [options.controller, getValues, validate, isValid]);

    return {
        getValues,
        getValue,
        setValues,
        setValue,
        setError,
        update,
        watch,
        validate,
        isValid,
        errors: errors.current,
    };
}
