import React from 'react';

import { FormErrorSchema } from 'shared/types/FormErrorSchema';

export interface UseFormControllerPub<T> {
    validate: () => Optional<boolean>;
    isValid: () => Optional<boolean>;
    getValues: () => Optional<T>;
    setError: <K extends keyof T>(key: K, errorList: Optional<FormErrorSchema>) => void;
}

export interface UseFormControllerSub<T> {
    on<K extends keyof UseFormControllerPub<T>>(name: K, callback: UseFormControllerPub<T>[K]): () => void;
}

export interface UseFormControllerResult<T> extends UseFormControllerPub<T> {
    controller: UseFormControllerSub<T>;
}

export function useFormController<
    T,
    K extends keyof UseFormControllerPub<T> = keyof UseFormControllerPub<T>,
>(): UseFormControllerResult<T> {
    const store = React.useRef<Partial<Record<keyof UseFormControllerPub<T>, UseFormControllerPub<T>[K]>>>({});

    const subscribe = React.useCallback(
        (name: K, callback: UseFormControllerPub<T>[K]) => {
            store.current[name] = callback;

            return () => {
                store.current[name] = undefined;
            };
        },
        [store],
    ) as <K extends keyof UseFormControllerPub<T>>(name: K, callback: UseFormControllerPub<T>[K]) => () => void;

    const trigger = React.useCallback(
        (name: K, args?: Parameters<UseFormControllerPub<T>[K]>) => {
            const callback = store.current[name];

            if (callback) {
                return callback.apply(null, args);
            }

            return undefined;
        },
        [store],
    ) as <K extends keyof UseFormControllerPub<T>>(
        name: K,
        args?: Parameters<UseFormControllerPub<T>[K]>,
    ) => Optional<ReturnType<UseFormControllerPub<T>[K]>>;

    const controller = React.useMemo(() => {
        return {
            on: subscribe,
        };
    }, [subscribe]);

    const isValid = React.useCallback(() => {
        return trigger('isValid');
    }, [trigger]);

    const validate = React.useCallback(() => {
        return trigger('validate');
    }, [trigger]);

    const getValues = React.useCallback(() => {
        return trigger('getValues');
    }, [trigger]);

    const setError = React.useCallback(
        (key, errorList) => {
            return trigger('setError', [key, errorList]);
        },
        [trigger],
    ) as <K extends keyof T>(key: K, errorList: Optional<FormErrorSchema>) => void;

    return {
        isValid,
        validate,
        getValues,
        setError,
        controller,
    };
}
