import * as React from 'react';

import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';

const TIMEOUT = 100;

export function useScrollSync(...refs: Array<{ current?: Nullable<HTMLElement> }>) {
    const timerRef = useTimerRef(0);

    const handleScroll = React.useCallback(
        ({ target }) => {
            const syncedRefs = refs.filter((ref) => ref.current !== target);
            const targetScrollLeft = target.scrollLeft;

            clearTimeout(timerRef.current);

            syncedRefs.forEach((ref) => {
                removeEvent(ref);

                if (ref.current) {
                    ref.current.style.willChange = 'scroll-position';
                    ref.current.scrollLeft = targetScrollLeft;
                }
            });

            timerRef.current = setTimeout(() => {
                syncedRefs.forEach((ref) => {
                    addEvent(ref);

                    if (ref.current) {
                        ref.current.style.willChange = '';
                    }
                });
            }, TIMEOUT) as unknown as number;
        },
        // eslint-disable-next-line react-hooks/exhaustive-deps
        [],
    );

    const addEvent = React.useCallback(
        (ref) => {
            ref.current.addEventListener('scroll', handleScroll, { passive: true });
        },
        [handleScroll],
    );

    const removeEvent = React.useCallback(
        (ref) => {
            ref.current.removeEventListener('scroll', handleScroll, { passive: true });
        },
        [handleScroll],
    );

    React.useLayoutEffect(() => {
        refs.forEach(addEvent);

        return () => {
            refs.forEach(removeEvent);
        };
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);
}
