import * as React from 'react';
import cn from 'classnames/bind';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';

import styles from 'shared/ui/Button/Button.css';

export type ButtonAttributes = (JSX.IntrinsicElements['a'] & JSX.IntrinsicElements['button']) & {
    'data-navigate'?: string;
    'data-original-href'?: string;
    'data-trigger'?: string;
};

export interface ButtonProps extends ButtonAttributes {
    className?: string;
    tagName?: keyof React.ReactHTML;

    color?: ButtonColor;
    size?: ButtonSize;

    icon?: SvgIcon;
    label?: React.ReactNode;

    disabled?: boolean;

    round?: boolean;
    shadow?: boolean;
    wide?: boolean;

    onClick?(event: React.MouseEvent<HTMLAnchorElement | HTMLButtonElement>): void;
}

const cx = cn.bind(styles);

export const Button = React.memo(
    React.forwardRef<HTMLElement, ButtonProps>(function Button(
        {
            className,
            tagName,
            color,
            size,
            icon: Icon,
            label,
            disabled,
            round,
            shadow,
            wide,
            onClick,
            children,
            ...otherProps
        },
        ref,
    ) {
        const TagName: string = otherProps.href ? 'a' : tagName || 'button';

        const attrs: ButtonAttributes = {
            ...otherProps,
            className: cx(
                styles.button,
                {
                    iconOnly: Boolean(!label && Icon),
                    disabled,
                    round,
                    shadow,
                    wide,
                },
                [color, size, className],
            ),
            title: typeof label === 'string' ? label : otherProps.title,
            onClick,
            ref: ref as React.ForwardedRef<any>,
        };

        return (
            <TagName {...attrs}>
                {Icon && (
                    <span className={styles.icon}>
                        <Icon />
                    </span>
                )}

                {label && <span className={styles.label}>{label}</span>}

                {children}
            </TagName>
        );
    }),
);
