import * as React from 'react';
import { useDatepicker } from '@datepicker-react/hooks';

import { createStorybookMeta } from 'tests/storybook/helpers/createStorybookMeta';
import { createStorybookTemplate } from 'tests/storybook/helpers/createStorybookTemplate';

import { getCalendarMonths } from 'shared/helpers/getCalendarMonths/getCalendarMonths';
import { Calendar, CalendarProps } from 'shared/ui/Calendar/Calendar';

const DATE = new Date('2022-01-12');

const storybookMeta = createStorybookMeta<CalendarProps>({
    title: 'shared/Calendar',
    component: Calendar,

    parameters: { date: DATE },

    hermioneSelector: '[data-portal-id] > div',

    args: {
        activeMonths: getCalendarMonths(DATE),
    },
});

export default storybookMeta;

async function hoverDate(browser: WebdriverIO.Browser, selector: string) {
    await browser.$('button[type=button]:nth-child(20)').moveTo({ xOffset: 10, yOffset: 10 });

    return browser.assertView('hover', selector);
}

interface ContextTypes {
    startDate: Nullable<Date>;
    endDate: Nullable<Date>;
}

const Template = createStorybookTemplate<CalendarProps & ContextTypes>(
    ({ activeMonths, startDate = null, endDate = null, ...otherProps }) => {
        const dayContext = useDatepicker({
            startDate,
            endDate,
            focusedInput: null,
            onDatesChange: () => {},
        });

        return (
            <Calendar
                {...otherProps}
                activeMonths={activeMonths}
                dayContext={dayContext}
            />
        );
    },
);

export const single = Template.bind({});
single.hermioneFn = hoverDate;

export const singleStartDate = Template.bind({});
singleStartDate.args = { startDate: new Date('2022-01-01') };
singleStartDate.hermioneFn = hoverDate;

export const singleEndDate = Template.bind({});
singleEndDate.args = { endDate: new Date('2022-01-31') };
singleEndDate.hermioneFn = hoverDate;

export const singleRange = Template.bind({});
singleRange.args = {
    startDate: new Date('2022-01-01'),
    endDate: new Date('2022-01-31'),
};
singleRange.hermioneFn = hoverDate;

export const double = Template.bind({});
double.args = { activeMonths: getCalendarMonths(DATE, 2) };

export const doubleRange = Template.bind({});
doubleRange.args = {
    activeMonths: getCalendarMonths(DATE, 2),
    startDate: new Date('2022-01-01'),
    endDate: new Date('2022-02-28'),
};

export const year = Template.bind({});
year.args = { backYearStep: 100 };
year.hermioneFn = async function (browser, selector) {
    const titleLink = '[class*="__title"] > [class*="__link"]';

    await browser.$(titleLink).moveTo({ xOffset: 10, yOffset: 10 });
    await browser.assertView('title-hover', selector);

    await browser.$(titleLink).click();
    await browser.$('body').moveTo({ xOffset: 1, yOffset: 1 });

    return browser.assertView('year', selector);
};
