import * as React from 'react';
import { MonthType } from '@datepicker-react/hooks/lib/useDatepicker/useDatepicker.utils';
import cn from 'classnames';

import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { CalendarMonth, CalendarMonthProps } from 'shared/ui/CalendarMonth/CalendarMonth';
import { CalendarYear, CalendarYearProps } from 'shared/ui/CalendarYear/CalendarYear';

import styles from 'shared/ui/Calendar/Calendar.css';

export interface CalendarProps
    extends Pick<CalendarMonthProps, 'dayContext' | 'onDayPick'>,
        Pick<CalendarYearProps, 'backYearStep' | 'nextYearStep'> {
    className?: string;
    style?: React.CSSProperties;

    activeMonths: Nullable<MonthType[]>;

    onPreviousClick(date: Date): void;

    onNextClick(date: Date): void;

    onYearPick(date: Date): void;
}

function getDateByYear(months: Nullable<MonthType[]>, year: number): Date {
    const date = months?.[0].date || new Date();

    return new Date(year, date.getMonth(), 1);
}

function getDateByDirection(months: Nullable<MonthType[]>, direction: number): Date {
    const date = months?.[0].date || new Date();

    return new Date(date.getFullYear(), date.getMonth() + direction, 1);
}

export const Calendar = React.forwardRef<HTMLDivElement, CalendarProps>(function Calendar(
    {
        className,
        style,
        activeMonths,
        backYearStep,
        nextYearStep,
        onPreviousClick,
        onNextClick,
        onYearPick,
        ...otherProps
    },
    ref,
) {
    const [isYearChecking, setYearChecking] = React.useState<boolean>();

    const Portal = usePortal();

    const onTitleClickHandler = React.useCallback(() => {
        setYearChecking(!isYearChecking);
    }, [isYearChecking]);

    const onPreviousClickHandler = React.useCallback(() => {
        onPreviousClick(getDateByDirection(activeMonths, -1));
    }, [activeMonths, onPreviousClick]);

    const onNextClickHandler = React.useCallback(() => {
        onNextClick(getDateByDirection(activeMonths, 1));
    }, [activeMonths, onNextClick]);

    const onYearPickHandler = React.useCallback(
        ({ currentTarget }: React.MouseEvent<HTMLLIElement>) => {
            const year = Number(currentTarget.getAttribute('data-value'));

            onYearPick(getDateByYear(activeMonths, year));

            onTitleClickHandler();
        },
        [activeMonths, onYearPick, onTitleClickHandler],
    );

    if (!activeMonths) {
        return null;
    }

    const hasYearChecking = Boolean(backYearStep || nextYearStep);
    const monthCount = activeMonths.length;

    return (
        <Portal>
            <div
                className={cn(styles.calendar, className)}
                style={{ ...style, '--months-count': monthCount } as React.CSSProperties}
                ref={ref}
            >
                {isYearChecking && (
                    <CalendarYear
                        className={styles.year}
                        checkedYear={activeMonths[0].year}
                        backYearStep={backYearStep}
                        nextYearStep={nextYearStep}
                        onTitleClick={onTitleClickHandler}
                        onYearPick={onYearPickHandler}
                    />
                )}

                {activeMonths.map(({ month, year }, index) => {
                    return (
                        <CalendarMonth
                            {...otherProps}
                            month={month}
                            year={year}
                            arrows={monthCount > 1 ? (index ? 'next' : 'previous') : undefined}
                            onTitleClick={hasYearChecking ? onTitleClickHandler : undefined}
                            onPreviousClick={onPreviousClickHandler}
                            onNextClick={onNextClickHandler}
                            key={`${year}-${month}`}
                        />
                    );
                })}
            </div>
        </Portal>
    );
});
