import * as React from 'react';
import { useDay } from '@datepicker-react/hooks';
import cn from 'classnames/bind';

import { isSameDate } from 'shared/helpers/isSameDate/isSameDate';

import styles from 'shared/ui/CalendarDay/CalendarDay.css';

export interface CalendarDayContext {
    focusedDate: Nullable<Date>;

    isDateFocused(date: Date): boolean;

    isDateSelected(date: Date): boolean;

    isDateHovered(date: Date): boolean;

    isDateBlocked(date: Date): boolean;

    isFirstOrLastSelectedDate(date: Date): boolean;

    onDateFocus(date: Date): void;

    onDateSelect(date: Date): void;

    onDateHover(date: Date): void;
}

export interface CalendarDayProps {
    className?: string;
    date: Date;
    dayContext?: CalendarDayContext;

    onDayPick(date: Date): void;
}

const cx = cn.bind(styles);

const DEFAULT_CONTEXT: CalendarDayContext = {
    focusedDate: null,
    isDateFocused: () => false,
    isDateSelected: () => false,
    isDateHovered: () => false,
    isDateBlocked: () => false,
    isFirstOrLastSelectedDate: () => false,
    onDateFocus: () => {},
    onDateSelect: () => {},
    onDateHover: () => {},
};

export const CalendarDay: React.FC<CalendarDayProps> = function CalendarDay({
    className,
    date,
    dayContext,
    onDayPick,
}) {
    const dayRef = React.useRef<Nullable<HTMLButtonElement>>(null);

    const {
        isSelected,
        isSelectedStartOrEnd,
        isWithinHoverRange,
        disabledDate,
        onClick,
        onKeyDown,
        onMouseEnter,
        tabIndex,
    } = useDay({
        date,
        dayRef,
        ...DEFAULT_CONTEXT,
        ...dayContext,
    });

    const onClickHandler = React.useCallback(() => {
        onClick();

        if (!disabledDate) {
            onDayPick(date);
        }
    }, [disabledDate, onDayPick, onClick]);

    const isCurrentDay = isSameDate(new Date(), date);

    return (
        <button
            className={cx(
                styles.calendarDay,
                {
                    current: isCurrentDay,
                    selected: isSelected || isSelectedStartOrEnd || isWithinHoverRange,
                    selectedCorner: isSelectedStartOrEnd,
                    disabled: disabledDate,
                },
                [className],
            )}
            type="button"
            onClick={onClickHandler}
            onKeyDown={onKeyDown}
            onMouseEnter={onMouseEnter}
            ref={dayRef}
            tabIndex={tabIndex}
        >
            <span className={styles.label}>{date.getDate()}</span>
        </button>
    );
};
