import * as React from 'react';
import { useMonth } from '@datepicker-react/hooks';
import cn from 'classnames';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { ButtonSize } from 'shared/consts/ButtonSize';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { Button } from 'shared/ui/Button/Button';
import { CalendarDay, CalendarDayProps } from 'shared/ui/CalendarDay/CalendarDay';
import { Link } from 'shared/ui/Link/Link';

import ArrowIcon from 'shared/ui/Icons/images/arrow-short-forward-24.inline.svg';

import styles from 'shared/ui/CalendarMonth/CalendarMonth.css';

export interface CalendarMonthProps extends Pick<CalendarDayProps, 'dayContext' | 'onDayPick'> {
    className?: string;
    month: number;
    year: number;
    arrows?: 'previous' | 'next';

    onTitleClick?(): void;

    onPreviousClick(): void;

    onNextClick(): void;
}

export const CalendarMonth: React.FC<CalendarMonthProps> = function CalendarMonth({
    className,
    month,
    year,
    arrows,
    onTitleClick,
    onPreviousClick,
    onNextClick,
    ...otherProps
}) {
    const { days } = useMonth({
        month,
        year,
    });

    const locale = getI18nLocale();

    return (
        <div className={cn(styles.calendarMonth, className)}>
            <div className={styles.header}>
                {(!arrows || arrows === 'previous') && (
                    <Button
                        className={styles.arrow}
                        size={ButtonSize.M}
                        icon={ArrowIcon}
                        onClick={onPreviousClick}
                    />
                )}

                <h5 className={styles.title}>
                    <Link
                        className={styles.titleLink}
                        disabled={!onTitleClick}
                        onClick={onTitleClick}
                    >
                        {getDateString(new Date(year, month, 1), locale, { month: 'long' })}, {year}
                    </Link>
                </h5>

                {(!arrows || arrows === 'next') && (
                    <Button
                        className={styles.arrow}
                        size={ButtonSize.M}
                        icon={ArrowIcon}
                        onClick={onNextClick}
                    />
                )}
            </div>

            <div className={styles.body}>
                {days.map((day, index) => {
                    if (typeof day === 'object') {
                        return (
                            <CalendarDay
                                {...otherProps}
                                date={day.date}
                                key={day.dayLabel}
                            />
                        );
                    }

                    return <span key={index} />;
                })}
            </div>
        </div>
    );
};
