import * as React from 'react';
import cn from 'classnames';

import { MenuType } from 'shared/consts/MenuType';
import { Link } from 'shared/ui/Link/Link';
import { Menu } from 'shared/ui/Menu/Menu';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import { i18n } from 'shared/ui/CalendarYear/CalendarYear.i18n';

import styles from 'shared/ui/CalendarYear/CalendarYear.css';

export interface CalendarYearProps {
    className?: string;
    checkedYear?: number;
    backYearStep?: number;
    nextYearStep?: number;

    onTitleClick(): void;

    onYearPick(event: React.MouseEvent<HTMLLIElement>): void;
}

export const CalendarYear: React.FC<CalendarYearProps> = function CalendarYear({
    className,
    checkedYear,
    backYearStep,
    nextYearStep,
    onTitleClick,
    onYearPick,
}) {
    const year = new Date().getFullYear();

    const items = React.useMemo((): MenuItemOptions[] => {
        let res: MenuItemOptions[] = [];
        let keys = -1;

        const fromYear = nextYearStep ? year + nextYearStep : year;

        if (backYearStep || nextYearStep) {
            keys = (backYearStep ?? 0) + (nextYearStep ?? 0);
        }

        for (let i = 0; i <= keys; i++) {
            const value = (fromYear - i).toString();

            res.push({
                label: value,
                value,
            });
        }

        if (!backYearStep) {
            return res.reverse();
        }

        return res;
    }, [backYearStep, nextYearStep, year]);

    if (!items || !items.length) {
        return null;
    }

    return (
        <div className={cn(styles.calendarYear, className)}>
            <h4 className={styles.title}>
                <Link
                    className={styles.titleLink}
                    onClick={onTitleClick}
                >
                    {i18n('Choose year')}
                </Link>
            </h4>

            <Menu
                className={styles.menu}
                items={items}
                menuType={MenuType.RADIO}
                visibleCheck={false}
                checked={checkedYear?.toString()}
                onItemClick={onYearPick}
            />
        </div>
    );
};
