import * as React from 'react';
import cn from 'classnames/bind';

import styles from 'shared/ui/ChartContainer/ChartContainer.css';

const cx = cn.bind(styles);

const DEFAULT_CHART_WIDTH = 480;
const DEFAULT_CHART_HEIGHT = 300;

export interface ChartContainerProps {
    className?: string;

    width?: number;
    height?: number;
    visible?: boolean;
    hasAxis?: boolean;
    legend?: React.ReactNode;

    onMouseMove?(event: React.MouseEvent<SVGSVGElement>): void;

    onMouseLeave?(event: React.MouseEvent<SVGSVGElement>): void;

    setContainerRef?: ((element: Nullable<HTMLDivElement>) => void) | React.MutableRefObject<Nullable<HTMLDivElement>>;
    setSVGRef?: ((element: Nullable<SVGSVGElement>) => void) | React.MutableRefObject<Nullable<SVGSVGElement>>;
    setXAxisRef?: ((element: Nullable<SVGGElement>) => void) | React.MutableRefObject<Nullable<SVGGElement>>;
    setYAxisRef?: ((element: Nullable<SVGGElement>) => void) | React.MutableRefObject<Nullable<SVGGElement>>;
    setYCustomAxisRef?: ((element: Nullable<SVGGElement>) => void) | React.MutableRefObject<Nullable<SVGGElement>>;
    setLeftAxisSignatureRef?:
        | ((element: Nullable<SVGGElement>) => void)
        | React.MutableRefObject<Nullable<SVGGElement>>;
    setRightAxisSignatureRef?:
        | ((element: Nullable<SVGGElement>) => void)
        | React.MutableRefObject<Nullable<SVGGElement>>;
}

export const ChartContainer: React.FC<ChartContainerProps> = function ChartContainer({
    className,
    width = DEFAULT_CHART_WIDTH,
    height = DEFAULT_CHART_HEIGHT,
    visible = true,
    hasAxis = true,
    legend,
    onMouseMove,
    onMouseLeave,
    setContainerRef,
    setSVGRef,
    setXAxisRef,
    setYAxisRef,
    setYCustomAxisRef,
    setLeftAxisSignatureRef,
    setRightAxisSignatureRef,
    children,
    ...otherProps
}) {
    return (
        <div
            className={cx(styles.container, { visible }, [className])}
            ref={setContainerRef}
        >
            <svg
                className={cx(styles.svg)}
                viewBox={`0 0 ${width} ${height}`}
                preserveAspectRatio="xMinYMin meet"
                onMouseMove={onMouseMove}
                onMouseLeave={onMouseLeave}
                ref={setSVGRef}
            >
                {hasAxis && (
                    <>
                        <g
                            className={cx(styles.axisSignature, { left: true })}
                            ref={setLeftAxisSignatureRef}
                        />
                        <g
                            className={cx(styles.axisSignature, { right: true })}
                            transform={`translate(${width}, 0)`}
                            ref={setRightAxisSignatureRef}
                        />
                        <g
                            className={cx(styles.axis, { y: true })}
                            ref={setYAxisRef}
                        />
                        <g
                            className={cx(styles.axis, { custom: true })}
                            ref={setYCustomAxisRef}
                        />
                    </>
                )}
                {children}
                {hasAxis && (
                    <>
                        <g
                            className={cx(styles.axis, { x: true })}
                            ref={setXAxisRef}
                        />
                    </>
                )}
            </svg>

            {legend}
        </div>
    );
};
