import * as React from 'react';
import cn from 'classnames/bind';

import { formatChartValue } from 'shared/helpers/formatChartValue/formatChartValue';
import { formatDateToString } from 'shared/helpers/formatDateToString/formatDateToString';
import { getChartArrayPoints } from 'shared/helpers/getChartArrayPoints/getChartArrayPoints';
import { ChartKeysFormat } from 'shared/types/ChartKeysFormat';
import { ChartKey, ChartPoint } from 'shared/types/ChartPoint';

import ArrowIcon from 'shared/ui/Icons/images/popup-arrow.inline.svg';

import styles from 'shared/ui/ChartLegendPopup/ChartLegendPopup.css';

const cx = cn.bind(styles);

function renderLegendLines(point: ChartPoint, keys: ChartKey[], keysFormat: ChartKeysFormat[], legend?: string[]) {
    return keys.map((key, keyIndex) => {
        let values = getChartArrayPoints(point, key);

        return values.map((value, valueIndex) => {
            let legendIndex = values.length > 1 ? valueIndex : keyIndex;

            return (
                <li
                    className={cx(styles.line)}
                    key={key + value}
                >
                    <span className={cx(styles.value)}>
                        {formatChartValue({ value: value!, ...keysFormat[keyIndex] })}
                    </span>
                    {legend && <span className={cx(styles.label)}>{legend[legendIndex]}</span>}
                </li>
            );
        });
    });
}

export interface ChartLegendPopupProps {
    className?: string;
    points: Nullable<ChartPoint> | ChartPoint[];
    keys: ChartKey[];
    keysFormat?: ChartKeysFormat[];
    legend?: string[];
    arrowPosition?: 'left' | 'right';
    style?: React.CSSProperties;
    hideDate?: boolean;

    setRef?: ((element: Nullable<HTMLDivElement>) => void) | React.MutableRefObject<Nullable<HTMLDivElement>>;
}

export const ChartLegendPopup: React.FC<ChartLegendPopupProps> = function ChartLegendPopup({
    className,
    points,
    keys,
    keysFormat = [],
    legend,
    arrowPosition,
    style,
    setRef,
    hideDate,
    ...otherProps
}) {
    if (!points) {
        return null;
    }

    let arrayOfPoints = Array.isArray(points) ? points : [points];

    if (!arrayOfPoints.length) {
        return null;
    }

    return (
        <div
            className={cx(styles.container, [arrowPosition], [className])}
            style={style}
            ref={setRef}
        >
            {Boolean(arrowPosition) && <ArrowIcon className={cx(styles.arrow)} />}
            {arrayOfPoints.map((point, index) => {
                let date = point.begin_date;

                return (
                    <div
                        className={cx(styles.item)}
                        key={date + index}
                    >
                        {!hideDate && <span className={cx(styles.date)}>{formatDateToString(date)}</span>}
                        <ul className={cx(styles.lines)}>{renderLegendLines(point, keys, keysFormat, legend)}</ul>
                    </div>
                );
            })}
        </div>
    );
};
