import * as React from 'react';
import cn from 'classnames/bind';
import { ScaleLinear, ScaleTime } from 'd3-scale';

import {
    CHART_AXIS_MARGIN_BOTTOM,
    CHART_MARGIN_TOP,
    DEFAULT_CHART_HEIGHT,
    DEFAULT_CHART_WIDTH,
} from 'shared/consts/Chart';
import { ChartViewHistogramSize } from 'shared/consts/ChartViewHistogramSize';
import { addEventListener } from 'shared/helpers/addEventListener/addEventListener';
import { d3Max } from 'shared/helpers/d3Max/d3Max';
import { d3ScaleTime } from 'shared/helpers/d3ScaleTime/d3ScaleTime';
import { d3ScaleTimeLinear } from 'shared/helpers/d3ScaleTimeLinear/d3ScaleTimeLinear';
import { d3ScaleY } from 'shared/helpers/d3ScaleY/d3ScaleY';
import { d3SelectDays } from 'shared/helpers/d3SelectDays/d3SelectDays';
import { d3SelectDottedLines } from 'shared/helpers/d3SelectDottedLines/d3SelectDottedLines';
import { d3SelectLines } from 'shared/helpers/d3SelectLines/d3SelectLines';
import { d3SelectNumbers } from 'shared/helpers/d3SelectNumbers/d3SelectNumbers';
import { d3SelectSignatures } from 'shared/helpers/d3SelectSignatures/d3SelectSignatures';
import { getChartArrayPoints } from 'shared/helpers/getChartArrayPoints/getChartArrayPoints';
import { getChartClosePoint } from 'shared/helpers/getChartClosePoint/getChartClosePoint';
import { getChartFilledPoints } from 'shared/helpers/getChartFilledPoints/getChartFilledPoints';
import { getChartRangeValue } from 'shared/helpers/getChartRangeValue/getChartRangeValue';
import { DateAggregationConfig } from 'shared/helpers/getDateAggregation/getDateAggregation';
import { ChartPoint } from 'shared/types/ChartPoint';
import { ChartViewOptions } from 'shared/types/ChartViewOptions';
import { ChartContainer } from 'shared/ui/ChartContainer/ChartContainer';
import { ChartLegendPopup } from 'shared/ui/ChartLegendPopup/ChartLegendPopup';

import { i18n } from 'shared/ui/ChartViewHistogram/ChartViewHistogram.i18n';

import styles from 'shared/ui/ChartViewHistogram/ChartViewHistogram.css';

const AXIS_MARGIN = { M: 64 };
const MARGIN_RIGHT = { M: 24 };
const BAR_WIDTH_MAX = { M: 28 };
const BAR_WIDTH_MIN = { M: 6 };
const BAR_WIDTH_MAX_POINTS = 12;
const BAR_SHIFT = { M: 2 };
const BAR_RADIUS = 2;
const LEGEND_GAP_X_MIN = { M: 12 };
const LEGEND_GAP_Y = 6;

const cx = cn.bind(styles);

export enum ChartViewType {
    DATE = 'date',
    VALUE = 'value',
}

export interface ChartViewHistogramProps extends ChartViewOptions {
    className?: string;
    size?: ChartViewHistogramSize;
    aggregationConfig: DateAggregationConfig;
    barColors?: string[];
    viewType?: ChartViewType;
    dottedLine?: number;
}

export const ChartViewHistogram: React.FC<ChartViewHistogramProps> = function ChartViewHistogram({
    className,
    size = ChartViewHistogramSize.M,
    points,
    keys,
    keysFormat = [],
    aggregationConfig,
    legend,
    width = DEFAULT_CHART_WIDTH,
    height = DEFAULT_CHART_HEIGHT,
    barColors,
    viewType = ChartViewType.DATE,
    dottedLine,
    ...otherProps
}) {
    let containerRef = React.useRef<Nullable<HTMLDivElement>>(null);
    let svgRef = React.useRef<Nullable<SVGSVGElement>>(null);

    let xAxisRef = React.useRef<Nullable<SVGGElement>>(null);
    let yAxisRef = React.useRef<Nullable<SVGGElement>>(null);
    let yCustomAxisRef = React.useRef<Nullable<SVGGElement>>(null);

    let leftAxisSignatureRef = React.useRef<Nullable<SVGGElement>>(null);
    let rightAxisSignatureRef = React.useRef<Nullable<SVGGElement>>(null);

    let legendRef = React.useRef<Nullable<HTMLDivElement>>(null);
    let legendArrowPosition = React.useRef<'left' | 'right'>('left');

    let [chartSize, setChartSize] = React.useState<{ width: number; height: number }>({ width, height });
    let [visible, setVisible] = React.useState<boolean>(false);
    let [hoveredPoints, setHoveredPoints] = React.useState<ChartPoint[]>([]);
    let [legendShift, setLegendShift] = React.useState<number[]>([0, 0]);

    let chartPoints = React.useMemo(() => {
        return viewType === ChartViewType.DATE ? getChartFilledPoints(points, aggregationConfig) : points;
    }, [viewType, points, aggregationConfig]);

    let onResize = React.useCallback(() => {
        setVisible(false);

        if (containerRef.current) {
            let { width, height } = containerRef.current.getBoundingClientRect();

            setChartSize({ width, height });
            setVisible(true);
        }
    }, []);

    React.useEffect(() => {
        onResize();
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);

    React.useEffect(() => {
        return addEventListener(window, 'resize', onResize, { passive: true });
    }, [onResize]);

    // axis: calculate parts
    let xScale = React.useMemo(() => {
        if (ChartViewType.VALUE === viewType) {
            return d3ScaleTimeLinear(chartPoints, chartSize.width, AXIS_MARGIN[size], MARGIN_RIGHT[size]);
        } else {
            return d3ScaleTime(chartPoints, chartSize.width, AXIS_MARGIN[size], MARGIN_RIGHT[size]);
        }
    }, [viewType, chartPoints, chartSize.width, size]);

    let yScale = React.useMemo(() => d3ScaleY(chartSize.height), [chartSize]);

    let yPointScale = React.useMemo(() => {
        let key = keys[0];
        let { postfix, maxValue } = keysFormat[0] || {};
        let maxYValue = postfix === '%' ? 100 : maxValue;

        let { from, to } = getChartRangeValue(chartPoints, key, maxYValue);

        return d3ScaleY(chartSize.height, [from, to]);
    }, [keys, keysFormat, chartPoints, chartSize.height]);

    // xAxis: draw days
    React.useEffect(() => {
        if (ChartViewType.VALUE === viewType) {
            d3SelectNumbers(chartPoints, xScale as ScaleLinear<number, number>, chartSize.height, xAxisRef.current);
        } else {
            d3SelectDays(chartPoints, xScale as ScaleTime<number, number>, chartSize.height, xAxisRef.current);
        }
    }, [xScale, chartSize, chartPoints, viewType]);

    // xAxis: draw signatures
    React.useEffect(
        () =>
            d3SelectSignatures({
                points: chartPoints,
                keys,
                keysFormat,
                height: chartSize.height,
                leftAxisElement: leftAxisSignatureRef.current,
                rightAxisElement: rightAxisSignatureRef.current,
            }),
        [chartSize, keys, keysFormat, chartPoints],
    );

    // yAxis: draw lines
    React.useEffect(() => {
        d3SelectLines(yScale, chartSize.width, yAxisRef.current);
    }, [yScale, chartSize]);

    // yAxis: draw dotted line
    React.useEffect(() => {
        if (dottedLine) {
            d3SelectDottedLines(dottedLine, '#241fff', chartSize.width, chartSize.height, yCustomAxisRef.current);
        }
    }, [chartSize, dottedLine]);

    const barWidth = React.useMemo(() => {
        const chartWidthScale = chartSize.width / DEFAULT_CHART_WIDTH;

        return (
            Math.max(
                BAR_WIDTH_MIN[size],
                Math.min(BAR_WIDTH_MAX[size], BAR_WIDTH_MAX[size] - chartPoints.length + BAR_WIDTH_MAX_POINTS),
            ) * chartWidthScale
        );
    }, [chartPoints.length, chartSize.width, size]);

    React.useEffect(() => {
        if (hoveredPoints.length && legendRef.current) {
            let { width, height } = legendRef.current.getBoundingClientRect();

            let key = keys[0];
            let posY = yPointScale(d3Max(hoveredPoints, key) || 0);
            let posX = xScale(hoveredPoints[0].begin_date);
            let gap = Math.max(LEGEND_GAP_X_MIN[size], barWidth);
            let baseLegendX = posX - gap - width;

            let shift: number[] = [];

            if (baseLegendX > AXIS_MARGIN[size]) {
                shift[0] = baseLegendX;
                legendArrowPosition.current = 'right';
            } else {
                shift[0] = posX + gap;
                legendArrowPosition.current = 'left';
            }

            shift[1] = posY - height / 2 + LEGEND_GAP_Y;

            setLegendShift(shift);
        }
    }, [xScale, yPointScale, keys, hoveredPoints, size, barWidth]);

    let charts = React.useMemo(() => {
        if (!chartPoints.length) {
            return [];
        }

        let chart: Array<{ xPos: number; date: number; values: number[] }> = [];

        keys.map((key) => {
            chartPoints.map((point, index) => {
                let values = getChartArrayPoints(point, key);
                let value = yPointScale(values[0] || 0);

                if (!chart[index]) {
                    let date = point.begin_date; // - weekShift;
                    let xPos = xScale(date);

                    chart[index] = { xPos, date, values: [value] };
                } else {
                    chart[index].values.push(value);
                }
            });
        });

        return chart;
    }, [xScale, yPointScale, keys, chartPoints]);

    let onMouseLeaveHandler = React.useCallback(() => {
        setHoveredPoints([]);
    }, []);

    let onMouseMoveHandler = React.useCallback(
        ({ clientX, clientY }: React.MouseEvent) => {
            if (!svgRef.current) {
                return;
            }

            let svgRect = svgRef.current.getBoundingClientRect();
            let { top, right, bottom, left } = svgRect;

            let inverted = xScale.invert(clientX - svgRect.x);
            let invertedTimestamp =
                ChartViewType.VALUE === viewType ? (inverted as number) : (inverted as Date).getTime();

            if (chartPoints.length) {
                if (
                    clientX >= left + AXIS_MARGIN[size] - barWidth / 2 - BAR_SHIFT[size] &&
                    clientX <= right - MARGIN_RIGHT[size] + barWidth / 2 + BAR_SHIFT[size] &&
                    clientY >= top + CHART_MARGIN_TOP &&
                    clientY <= bottom - CHART_AXIS_MARGIN_BOTTOM
                ) {
                    let closestPoint = getChartClosePoint(chartPoints, invertedTimestamp);

                    if (closestPoint) {
                        setHoveredPoints([closestPoint]);
                    }
                } else {
                    onMouseLeaveHandler();
                }
            }
        },
        [xScale, viewType, chartPoints, size, barWidth, onMouseLeaveHandler],
    );

    let { width: chartWidth, height: chartHeight } = chartSize;

    if (chartPoints.length === 0) {
        return <div className={styles.noData}>{i18n('There is no data')}</div>;
    }

    return (
        <ChartContainer
            className={cx(styles.chart, { hovered: Boolean(hoveredPoints.length) }, [className])}
            width={chartWidth}
            height={chartHeight}
            visible={visible}
            legend={
                <ChartLegendPopup
                    className={cx(styles.legend)}
                    points={hoveredPoints}
                    keys={keys}
                    keysFormat={keysFormat}
                    legend={legend}
                    arrowPosition={legendArrowPosition.current}
                    hideDate={viewType === ChartViewType.VALUE}
                    style={{
                        transform: `translate(${Math.round(legendShift[0])}px, ${legendShift[1]}px)`,
                    }}
                    setRef={legendRef}
                />
            }
            onMouseMove={onMouseMoveHandler}
            onMouseLeave={onMouseLeaveHandler}
            setContainerRef={containerRef}
            setSVGRef={svgRef}
            setXAxisRef={xAxisRef}
            setYAxisRef={yAxisRef}
            setYCustomAxisRef={yCustomAxisRef}
            setLeftAxisSignatureRef={leftAxisSignatureRef}
            setRightAxisSignatureRef={rightAxisSignatureRef}
        >
            {charts.map(({ xPos, date, values }, chartIndex) => (
                <g
                    className={cx(styles.bars, { current: hoveredPoints[0]?.begin_date === date })}
                    key={date}
                    data-testid="ChartViewHistogram.bars"
                >
                    {values.map((value, index) => {
                        let barShift = values.length > 1 ? [-BAR_SHIFT[size], BAR_SHIFT[size]] : [0];

                        return (
                            <rect
                                className={cx(styles.bar)}
                                transform={`translate(${xPos - barWidth / 2 + barShift[index]}, ${value + 1})`}
                                width={barWidth}
                                height={chartHeight - CHART_AXIS_MARGIN_BOTTOM - CHART_MARGIN_TOP - value}
                                rx={BAR_RADIUS}
                                ry={BAR_RADIUS}
                                key={date + index}
                                style={
                                    barColors
                                        ? {
                                              fill: barColors[chartIndex % barColors.length],
                                          }
                                        : {}
                                }
                            />
                        );
                    })}
                </g>
            ))}
        </ChartContainer>
    );
};
