import * as React from 'react';

import { Checkbox, CheckboxItemOptions, CheckboxProps } from 'shared/ui/Checkbox/Checkbox';
import { FilterGroup, FilterGroupProps } from 'shared/ui/FilterGroup/FilterGroup';

export interface CheckboxGroupProps extends Pick<FilterGroupProps, 'title'>, Pick<CheckboxProps, 'disabled'> {
    className?: string;

    items: CheckboxItemOptions[];
    values?: string[];

    onChange?(values: string[]): void;
}

export const CheckboxGroup: React.FC<CheckboxGroupProps> = React.memo(function CheckboxGroup({
    className,
    title,
    items,
    values,
    disabled,
    onChange,
    ...otherProps
}) {
    const onChangeHandler = React.useCallback(
        (checked: boolean, value: string) => {
            const valuesObj = items.map((item) => {
                const id = item.id.toString();

                return Object.assign({}, item, {
                    checked: id === value ? checked : values?.includes(id) || item.checked,
                });
            });

            if (onChange) {
                onChange(valuesObj.filter((value) => value.checked).map((value) => value.id.toString()));
            }
        },
        [items, values, onChange],
    );

    if (!items) {
        return null;
    }

    return (
        <FilterGroup
            className={className}
            title={title}
        >
            {items.map(({ id, label, disabled: itemDisabled }) => (
                <Checkbox
                    {...otherProps}
                    id={id}
                    label={label}
                    checked={values?.includes(id.toString())}
                    disabled={disabled || itemDisabled}
                    onChange={onChangeHandler}
                    key={id}
                />
            ))}
        </FilterGroup>
    );
});
