import * as React from 'react';
import { Placement } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { useFloatingMenu } from 'shared/hooks/useFloatingMenu/useFloatingMenu';
import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { ButtonArrow, ButtonArrowProps } from 'shared/ui/ButtonArrow/ButtonArrow';
import { Menu, MenuProps } from 'shared/ui/Menu/Menu';

import styles from 'shared/ui/Dropdown/Dropdown.css';

export interface DropdownProps
    extends ButtonArrowProps,
        Pick<MenuProps, 'items' | 'menuType' | 'visibleCheck' | 'selected'> {
    checkedMenu?: string | string[];
    position?: Placement;
    equalWidth?: boolean;
    maxHeight?: number;
    hasError?: boolean;

    'data-id'?: string;
    onMenuChange?(value: string | string[]): void;

    onMenuClick?(event: React.MouseEvent<HTMLLIElement>): void;
}

const cx = cn.bind(styles);

export const Dropdown: React.FC<DropdownProps> = function Dropdown({
    items,
    menuType,
    selected,
    checkedMenu,
    position = 'bottom-start',
    equalWidth,
    maxHeight,
    hasError,
    onClick,
    onFocus,
    onMenuChange,
    onMenuClick,
    ...otherProps
}) {
    const Portal = usePortal();

    const referenceRef = React.useRef() as React.MutableRefObject<Nullable<HTMLButtonElement>>;
    const floatingRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const { style, isMenuClosing, isMenuVisible, checked, onReferenceActionHandler, onMenuItemClickHandler } =
        useFloatingMenu<HTMLButtonElement>({
            referenceRef,
            floatingRef,
            value: checkedMenu,
            menuType,
            options: { placement: position, equalWidth },
            onReferenceClick: onClick,
            onReferenceFocus: onFocus,
            onMenuChange,
            onMenuClick,
        });

    return (
        <>
            <ButtonArrow
                {...otherProps}
                className={cx({ error: hasError }, [otherProps.className])}
                opened={isMenuVisible}
                onClick={onReferenceActionHandler}
                ref={referenceRef}
            />

            {isMenuVisible && items && (
                <Portal>
                    <Menu
                        className={cx(styles.menu, { close: isMenuClosing })}
                        shadow
                        style={{ ...style, maxHeight }}
                        items={items}
                        menuType={menuType}
                        checked={checked}
                        selected={selected}
                        onItemClick={onMenuItemClickHandler}
                        ref={floatingRef}
                    />
                </Portal>
            )}
        </>
    );
};
