import * as React from 'react';
import cn from 'classnames/bind';

import { addEventListener } from 'shared/helpers/addEventListener/addEventListener';
import { getScrollTop } from 'shared/helpers/getScrollTop/getScrollTop';
import { useMutationObserver } from 'shared/hooks/useMutationObserver/useMutationObserver';

import styles from 'shared/ui/FiltersContainer/FiltersContainer.css';

export interface FiltersContainerProps {
    className?: string;
    offsetTop?: number;

    clearButton?: React.ReactNode;
    isClearVisible?: boolean;
}

const cx = cn.bind(styles);

export const FiltersContainer: React.FC<FiltersContainerProps> = function FiltersContainer({
    className,
    offsetTop = 0,
    clearButton,
    isClearVisible,
    children,
}) {
    const ref = React.useRef<Nullable<HTMLDivElement>>(null);
    const position = React.useRef<{
        oldPosition: number;
        oldTime: number;
        newPosition: number;
        newTime: number;
        ticking: boolean;
    }>({
        oldPosition: 0,
        oldTime: 0,
        newPosition: 0,
        newTime: 0,
        ticking: false,
    });

    const onWindowScroll = React.useCallback(() => {
        position.current.newPosition = getScrollTop();
        position.current.newTime = Date.now();

        if (!position.current.ticking) {
            window.requestAnimationFrame(() => {
                translate();
                position.current.ticking = false;
                position.current.oldPosition = position.current.newPosition;
                position.current.oldTime = position.current.newTime;
            });

            position.current.ticking = true;
        }
    }, [position]);

    const translate = React.useCallback(() => {
        const container = ref.current;
        const parent = container?.parentElement;

        if (container && parent) {
            const { top, bottom } = parent.getBoundingClientRect();

            const distance = position.current.newPosition - position.current.oldPosition;
            const timeDelta = position.current.newTime - position.current.oldTime;

            // Hack for significant page height collapsing after filtering data
            // eslint-disable-next-line @typescript-eslint/no-magic-numbers
            if (timeDelta > 500 && distance < -100) {
                return;
            }

            // Don't scroll up before sticky period
            if (top - offsetTop > 0 && distance > 0) {
                return;
            }

            // Don't scroll down after sticky period
            if (bottom < window.innerHeight - offsetTop && distance < 0) {
                return;
            }

            container.scrollTop = container.scrollTop + distance;
        }
    }, [ref, position, offsetTop]);

    const onHeightChange = React.useCallback(() => {
        const container = ref.current;
        const parent = container?.parentElement;

        if (container && parent) {
            if (container.firstElementChild) {
                parent.style.minHeight = `${container.firstElementChild.clientHeight}px`;
            }
        }
    }, [ref]);

    React.useEffect(() => {
        const container = ref.current;

        if (container) {
            container.scrollTop = 0;

            onHeightChange();
        }
    }, []);

    React.useEffect(() => {
        addEventListener(window, 'scroll', onWindowScroll, { passive: true });
    }, [onWindowScroll]);

    useMutationObserver(ref, onHeightChange);

    const hasClear = Boolean(clearButton);

    return (
        <div
            className={cx(styles.filtersContainer, { hasClear }, [className])}
            ref={ref}
        >
            <div className={styles.wrap}>
                {children}

                {hasClear && <div className={cx(styles.clear, { visible: isClearVisible })}>{clearButton}</div>}
            </div>
        </div>
    );
};
