import * as React from 'react';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { InputPin } from 'shared/consts/InputPin';
import { InputSize } from 'shared/consts/InputSize';
import { formatTime } from 'shared/helpers/formatTime/formatTime';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { FormDateTimeRangeSchema } from 'shared/types/FormDateTimeRangeSchema';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { ErrorLabel } from 'shared/ui/ErrorLabel/ErrorLabel';
import { FormErrorContainer } from 'shared/ui/FormErrorContainer/FormErrorContainer';
import { FormLabel } from 'shared/ui/FormLabel/FormLabel';
import { Input } from 'shared/ui/Input/Input';
import { InputDateSingle, InputDateSingleProps } from 'shared/ui/InputDateSingle/InputDateSingle';

import { i18n } from 'shared/ui/FormDateTimeRange/FormDateTimeRange.i18n';

import styles from 'shared/ui/FormDateTimeRange/FormDateTimeRange.css';

export interface FormDateTimeRangeProps<T, K extends keyof T = keyof T>
    extends Pick<InputDateSingleProps, 'minBookingDate' | 'maxBookingDate'> {
    className?: string;
    title: string;

    actualSince?: Date;
    actualUntil?: Date;

    disableSince?: boolean;
    disableUntil?: boolean;

    update<K extends keyof T>(key: K): (value: T[K]) => void;

    watch<K extends keyof T>(key: K): Optional<T[K]>;

    readonly errors: Partial<Record<K, FormErrorSchema>>;

    sinceRef?: ((element: Nullable<HTMLElement>) => void) | React.MutableRefObject<Nullable<HTMLElement>>;
    untilRef?: ((element: Nullable<HTMLElement>) => void) | React.MutableRefObject<Nullable<HTMLElement>>;
}

const DATE_FORMAT: Intl.DateTimeFormatOptions = {
    day: '2-digit',
    month: '2-digit',
    year: 'numeric',
};

export function FormDateTimeRange<T extends OptionalRecord<FormDateTimeRangeSchema>>({
    className,
    title,
    actualSince,
    actualUntil,
    minBookingDate,
    maxBookingDate,
    disableSince,
    disableUntil,
    update,
    watch,
    errors,
    sinceRef,
    untilRef,
}: FormDateTimeRangeProps<T>) {
    const locale = getI18nLocale();

    return (
        <FormLabel
            className={className}
            title={title}
        >
            <div className={styles.items}>
                <FormErrorContainer
                    error={
                        actualSince ? (
                            <span className={styles.actualInfo}>
                                {i18n('Actually started {date} at {time}', {
                                    date: getDateString(actualSince, locale, DATE_FORMAT),
                                    time: formatTime(actualSince),
                                })}
                            </span>
                        ) : (
                            <ErrorLabel errors={errors.since || errors.since_time} />
                        )
                    }
                >
                    <div className={styles.item}>
                        <InputDateSingle
                            inputSize={InputSize.M}
                            pin={watch('since') ? InputPin.RIGHT : undefined}
                            placeholder={i18n('From')}
                            date={watch('since')}
                            minBookingDate={minBookingDate}
                            maxBookingDate={maxBookingDate}
                            disabled={disableSince}
                            hasError={Boolean(errors.since)}
                            onDateChange={update('since')}
                            setRef={sinceRef}
                        />

                        {watch('since') && (
                            <Input
                                className={styles.timeInput}
                                type="time"
                                inputSize={InputSize.M}
                                pin={InputPin.LEFT}
                                value={watch('since_time')}
                                disabled={disableSince}
                                hasError={Boolean(errors.since_time)}
                                onInputChange={update('since_time')}
                            />
                        )}
                    </div>
                </FormErrorContainer>

                <FormErrorContainer
                    error={
                        actualUntil ? (
                            <span className={styles.actualInfo}>
                                {i18n('Actually finished {date} at {time}', {
                                    date: getDateString(actualUntil, locale, DATE_FORMAT),
                                    time: formatTime(actualUntil),
                                })}
                            </span>
                        ) : (
                            <ErrorLabel errors={errors.until || errors.until_time} />
                        )
                    }
                >
                    <div className={styles.item}>
                        <InputDateSingle
                            inputSize={InputSize.M}
                            pin={watch('until') ? InputPin.RIGHT : undefined}
                            placeholder={i18n('Till')}
                            date={watch('until')}
                            minBookingDate={watch('since') || undefined}
                            maxBookingDate={maxBookingDate}
                            disabled={disableUntil}
                            hasError={Boolean(errors.until)}
                            onDateChange={update('until')}
                            setRef={untilRef}
                        />

                        {watch('until') && (
                            <Input
                                className={styles.timeInput}
                                type="time"
                                inputSize={InputSize.M}
                                pin={InputPin.LEFT}
                                value={watch('until_time')}
                                disabled={disableUntil}
                                hasError={Boolean(errors.until_time)}
                                onInputChange={update('until_time')}
                            />
                        )}
                    </div>
                </FormErrorContainer>
            </div>
        </FormLabel>
    );
}
