import * as React from 'react';
import cn from 'classnames';

import { isEqualArrays } from 'shared/helpers/isEqualArrays/isEqualArrays';
import { Button } from 'shared/ui/Button/Button';
import { ButtonFile, ButtonFileProps } from 'shared/ui/ButtonFile/ButtonFile';
import { Picture, PictureOptions } from 'shared/ui/Picture/Picture';

import PlusIcon from 'shared/ui/Icons/images/plus-24.inline.svg';
import TrashIcon from 'shared/ui/Icons/images/trash-outline-16.inline.svg';

import styles from 'shared/ui/ImagesPreviewUpload/ImagesPreviewUpload.css';

export interface ImagesPreviewUploadProps extends Pick<ButtonFileProps, 'multiple' | 'inputRef'> {
    className?: string;
    images?: PictureOptions[];
    files?: File[];
    maxLength?: number;
    isLoading?: boolean;

    onChange?(values: File[]): void;
}

export const ImagesPreviewUpload: React.FC<ImagesPreviewUploadProps> = React.memo(function ImagesPreviewUpload({
    className,
    multiple,
    images = [],
    files = [],
    maxLength,
    isLoading,
    onChange,
    inputRef,
    children,
}) {
    const [currentFiles, setCurrentFiles] = React.useState<File[]>(files);

    React.useEffect(() => {
        if (!isEqualArrays<File>(files, currentFiles)) {
            setCurrentFiles(files);
        }
    }, [files]);

    const items = React.useMemo(() => [...images, ...currentFiles], [images, currentFiles]);

    const onUploadHandler = React.useCallback(
        (values: File[]) => {
            const newFiles = [...currentFiles, ...values];

            setCurrentFiles(newFiles);

            if (onChange) {
                onChange(newFiles);
            }
        },
        [onChange, currentFiles],
    );

    const onRemoveHandler = React.useCallback(
        (event: React.MouseEvent<HTMLButtonElement>) => {
            let copyFiles = [...currentFiles];

            const index = event.currentTarget.getAttribute('data-index');

            if (index) {
                copyFiles.splice(Number(index), 1);

                setCurrentFiles(copyFiles);
            }

            if (onChange) {
                onChange(copyFiles);
            }
        },
        [onChange, currentFiles],
    );

    return (
        <div className={cn(styles.imagesPreview, className)}>
            {items?.map((item, index) => {
                const hasRemove = item instanceof File;

                let src: Optional<string>;
                let title;

                if (item instanceof File) {
                    src = URL.createObjectURL(item);
                    title = item.name;
                } else {
                    src = item.src;
                    title = item.title;
                }

                return (
                    <Picture
                        className={styles.picture}
                        src={src}
                        title={title}
                        isLoading={isLoading}
                        key={title ? title + index : index}
                    >
                        {hasRemove && (
                            <Button
                                className={styles.remove}
                                icon={TrashIcon}
                                onClick={onRemoveHandler}
                                data-index={index - items.length}
                            />
                        )}
                    </Picture>
                );
            })}

            {Boolean(onChange) && (maxLength ? items.length < maxLength : true) && (
                <ButtonFile
                    className={styles.button}
                    accept="image/*"
                    multiple={multiple}
                    icon={PlusIcon}
                    onInputChange={onUploadHandler}
                    inputRef={inputRef}
                />
            )}

            {children}
        </div>
    );
});
