import * as React from 'react';
import cn from 'classnames/bind';

import { InputPin } from 'shared/consts/InputPin';
import { InputSize } from 'shared/consts/InputSize';
import { KEY_BACKSPACE, KEY_DELETE, KEY_LEFT, KEY_RIGHT, KEY_SPACE, KEY_TAB } from 'shared/consts/keys';
import { useItemRef } from 'shared/hooks/useItemRef/useItemRef';
import { InputIcon } from 'shared/ui/InputIcon/InputIcon';

import CloseIcon16 from 'shared/ui/Icons/images/close-16.inline.svg';

import styles from 'shared/ui/Input/Input.css';

export type InputAttributes = JSX.IntrinsicElements['input'];

export interface InputProps extends InputAttributes {
    className?: string;
    style?: React.CSSProperties;
    inputSize?: InputSize;
    pin?: InputPin;

    focused?: boolean;
    disabled?: boolean;

    hasError?: boolean;
    hasClear?: boolean;

    hasAutoFocus?: boolean;
    shouldFocusOnClear?: boolean;

    withBackground?: boolean;

    onClear?(event: React.MouseEvent<HTMLSpanElement>): void;

    onInputChange?(value: string): void;

    setRef?: ((element: Nullable<HTMLElement>) => void) | React.MutableRefObject<Nullable<HTMLElement>>;
}

const cx = cn.bind(styles);

export const Input: React.FC<InputProps> = React.memo(function Input({
    className,
    style,
    type = 'text',
    inputMode = 'text',
    autoComplete = 'off',
    inputSize,
    pin,
    placeholder = '',
    value,
    focused,
    disabled,
    hasError,
    hasClear,
    hasAutoFocus,
    shouldFocusOnClear = true,
    withBackground,
    onChange,
    onClear,
    onKeyDown,
    onWheel,
    onInputChange,
    setRef,
    children,
    ...otherProps
}) {
    const { ref, itemRef } = useItemRef<HTMLElement>(setRef);

    React.useEffect(() => {
        if (hasAutoFocus && ref.current) {
            ref.current.focus();
        }
    }, [hasAutoFocus, ref.current]);

    const onClearHandler = React.useCallback(
        (event: React.MouseEvent<HTMLSpanElement>) => {
            if (onInputChange) {
                onInputChange('');
            }

            if (onClear) {
                onClear(event);
            }

            if (shouldFocusOnClear && ref.current) {
                ref.current.focus();
            }
        },
        [onInputChange, onClear, shouldFocusOnClear, ref.current],
    );

    const onChangeHandler = React.useCallback(
        (event: React.ChangeEvent<HTMLInputElement>) => {
            if (onInputChange) {
                onInputChange(event.target.value);
            }

            if (onChange) {
                onChange(event);
            }
        },
        [onInputChange, onChange],
    );

    const onInputNumberWheel = React.useCallback(
        (event: React.WheelEvent<HTMLInputElement>) => {
            const activeElem = document.activeElement as HTMLElement;

            if (activeElem && activeElem.tagName === 'INPUT' && (activeElem as HTMLInputElement).type === 'number') {
                activeElem.blur();
            }

            if (onWheel) {
                onWheel(event);
            }
        },
        [onWheel],
    );

    const onInputNumberKeyDown = React.useCallback(
        (event: React.KeyboardEvent<HTMLInputElement>) => {
            if (
                event.keyCode === KEY_SPACE ||
                (isNaN(Number(event.key)) &&
                    ![KEY_BACKSPACE, KEY_DELETE, KEY_LEFT, KEY_RIGHT, KEY_TAB].includes(event.keyCode))
            ) {
                event.preventDefault();
            }

            if (onKeyDown) {
                onKeyDown(event);
            }
        },
        [onKeyDown],
    );

    const isClearVisible = Boolean(!disabled && hasClear && value?.toString().length);
    const isTypeNumber = type === 'number';

    const attrs: InputAttributes = {
        ...otherProps,
        className: styles.inputInput,
        type: type || inputMode,
        inputMode,
        autoComplete,
        placeholder,
        value,
        disabled,
        onChange: onChangeHandler,
        ref: itemRef,
        onWheel: isTypeNumber ? onInputNumberWheel : onWheel,
        onKeyDown: isTypeNumber ? onInputNumberKeyDown : onKeyDown,
    };

    return (
        <div
            className={cx(
                styles.input,
                {
                    hasError,
                    hasClear,
                    focused,
                    disabled,
                    background: withBackground,
                },
                [inputSize, pin, className],
            )}
            style={style}
        >
            <input {...attrs} />

            {hasClear && (
                <InputIcon
                    className={styles.clear}
                    icon={CloseIcon16}
                    visible={isClearVisible}
                    onClick={onClearHandler}
                />
            )}

            {children}

            <span className={styles.border} />
        </div>
    );
});
