import * as React from 'react';
import { END_DATE, START_DATE, useDatepicker } from '@datepicker-react/hooks';
import { FocusedInput, OnDatesChangeProps } from '@datepicker-react/hooks/lib/useDatepicker/useDatepicker';
import { MonthType } from '@datepicker-react/hooks/lib/useDatepicker/useDatepicker.utils';
import { flip, Placement } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { getCalendarMonths } from 'shared/helpers/getCalendarMonths/getCalendarMonths';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { useClickOutside } from 'shared/hooks/useClickOutside/useClickOutside';
import { useFloatingUI } from 'shared/hooks/useFloatingUI/useFloatingUI';
import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';
import { Calendar, CalendarProps } from 'shared/ui/Calendar/Calendar';
import { InputProps } from 'shared/ui/Input/Input';
import { CLOSE_ANIMATION_TIME, DEFAULT_DATE_FORMAT, InputDate } from 'shared/ui/InputDate/InputDate';

import styles from 'shared/ui/InputDateRange/InputDateRange.css';

export interface InputDateRangeProps extends InputProps, Pick<CalendarProps, 'backYearStep' | 'nextYearStep'> {
    className?: string;

    placeholders?: string[];
    dates?: Nullable<Date>[];
    minBookingDate?: Date;
    maxBookingDate?: Date;
    dateFormat?: Intl.DateTimeFormatOptions;
    placement?: Placement;

    onDateChange?(values: [Optional<Date>, Optional<Date>]): void;
}

const cx = cn.bind(styles);
const CALENDAR_TYPES = [START_DATE, END_DATE];
const MONTHS_COUNT = 2;

export const InputDateRange: React.FC<InputDateRangeProps> = function InputDateRange({
    className,
    placeholders,
    dates,
    minBookingDate,
    maxBookingDate,
    backYearStep,
    nextYearStep,
    placement,
    dateFormat = DEFAULT_DATE_FORMAT,
    onClick,
    onClear,
    onDateChange,
    ...otherProps
}) {
    const locale = getI18nLocale();

    const containerRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
    const calendarRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    let startDateInputRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
    let endDateInputRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const [startDate, setStartDate] = React.useState<Nullable<Date>>(null);
    const [endDate, setEndDate] = React.useState<Nullable<Date>>(null);
    const [focusedInput, setFocusedInput] = React.useState<FocusedInput>(null);
    const [activeMonths, setActiveMonths] = React.useState<Nullable<MonthType[]>>(null);
    const [isCalendarClosing, setCalendarClosing] = React.useState<boolean>(false);
    const [isCalendarVisible, setCalendarVisible] = React.useState<boolean>(false);

    const { style, update } = useFloatingUI(containerRef, calendarRef, isCalendarVisible, {
        placement: placement || 'bottom-start',
        middleware: [flip({ crossAxis: false, padding: 8 })],
    });
    const timerRef = useTimerRef(0);

    React.useEffect(() => {
        setStartDate(dates?.length ? dates[0] : null);
        setEndDate(dates?.length ? dates[1] : null);
    }, [dates]);

    const onCalendarCloseHandler = React.useCallback(() => {
        if (!isCalendarClosing) {
            setCalendarClosing(true);
            setFocusedInput(null);

            timerRef.current = setTimeout(() => {
                setCalendarClosing(false);
                setCalendarVisible(false);
            }, CLOSE_ANIMATION_TIME) as unknown as number;
        }
    }, [timerRef, isCalendarClosing]);

    const onOutsideClickHandler = React.useCallback(
        ({ target }: MouseEvent) => {
            if (
                !startDateInputRef.current?.contains(target as Node) &&
                !endDateInputRef.current?.contains(target as Node)
            ) {
                onCalendarCloseHandler();
            }
        },
        [onCalendarCloseHandler],
    );

    const onDatesChangeHandler = React.useCallback(
        ({ startDate, endDate, focusedInput }: OnDatesChangeProps) => {
            setStartDate(startDate);
            setEndDate(endDate);
            setFocusedInput(focusedInput || START_DATE);
            setActiveMonths(getCalendarMonths(startDate, MONTHS_COUNT));

            if (!focusedInput && startDate && endDate) {
                onCalendarCloseHandler();
            }

            if (onDateChange) {
                onDateChange([startDate ? startDate : undefined, endDate ? endDate : undefined]);
            }
        },
        [onDateChange, onCalendarCloseHandler],
    );

    const onInputClickHandler = React.useCallback(
        (event: React.MouseEvent<HTMLInputElement>) => {
            const type = event.currentTarget.getAttribute('name') as FocusedInput;

            setFocusedInput(type);
            setActiveMonths(getCalendarMonths(startDate, MONTHS_COUNT));
            setCalendarVisible(true);
            update();

            if (onClick) {
                onClick(event);
            }
        },
        [onClick, update, startDate],
    );

    const onInputClearHandler = React.useCallback(
        (event: React.MouseEvent<HTMLSpanElement>) => {
            const type = event.currentTarget.parentNode?.firstElementChild?.getAttribute('name') as FocusedInput;
            const fromDate = type === START_DATE ? null : startDate;
            const tillDate = type === END_DATE ? null : endDate;

            setCalendarVisible(true);

            onDatesChangeHandler({ startDate: fromDate, endDate: tillDate, focusedInput: type });

            if (onClear) {
                onClear(event);
            }
        },
        [onClear, onDatesChangeHandler, startDate, endDate],
    );

    const onDayPickHandler = React.useCallback(
        (date: Date) => {
            if (focusedInput === END_DATE && !startDate) {
                onDatesChangeHandler({ startDate: null, endDate: date, focusedInput });
            }
        },
        [focusedInput, startDate, onDatesChangeHandler],
    );

    const onPreviousClickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date, MONTHS_COUNT));
    }, []);

    const onNextClickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date, MONTHS_COUNT));
    }, []);

    const onYearPickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date, MONTHS_COUNT));
    }, []);

    useClickOutside(calendarRef, onOutsideClickHandler);

    const dayContext = useDatepicker({
        startDate: startDate,
        endDate: endDate,
        minBookingDate,
        maxBookingDate,
        focusedInput,
        onDatesChange: onDatesChangeHandler,
    });

    return (
        <div
            className={className}
            ref={containerRef}
        >
            {CALENDAR_TYPES.map((type, index) => {
                const date = type === START_DATE ? startDate : endDate;
                const setRef = type === START_DATE ? startDateInputRef : endDateInputRef;

                return (
                    <InputDate
                        {...otherProps}
                        className={styles.inputInput}
                        name={type}
                        placeholder={placeholders && placeholders[index]}
                        value={date ? getDateString(date, locale, dateFormat) : ''}
                        focused={focusedInput === type}
                        onClick={onInputClickHandler}
                        onClear={onInputClearHandler}
                        setRef={setRef}
                        key={type}
                    />
                );
            })}

            {isCalendarVisible && (
                <Calendar
                    className={cx(styles.calendar, { close: isCalendarClosing })}
                    style={style}
                    activeMonths={activeMonths}
                    dayContext={dayContext}
                    backYearStep={backYearStep}
                    nextYearStep={nextYearStep}
                    onPreviousClick={onPreviousClickHandler}
                    onNextClick={onNextClickHandler}
                    onYearPick={onYearPickHandler}
                    onDayPick={onDayPickHandler}
                    ref={calendarRef}
                />
            )}
        </div>
    );
};
