import * as React from 'react';
import { useState } from 'react';
import { START_DATE, useDatepicker } from '@datepicker-react/hooks';
import { OnDatesChangeProps } from '@datepicker-react/hooks/lib/useDatepicker/useDatepicker';
import { MonthType } from '@datepicker-react/hooks/lib/useDatepicker/useDatepicker.utils';
import { flip } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { getI18nLocale } from 'utils/language/getI18nLocale';

import { getCalendarMonths } from 'shared/helpers/getCalendarMonths/getCalendarMonths';
import { getDateString } from 'shared/helpers/getDateString/getDateString';
import { useClickOutside } from 'shared/hooks/useClickOutside/useClickOutside';
import { useFloatingUI } from 'shared/hooks/useFloatingUI/useFloatingUI';
import { useItemRef } from 'shared/hooks/useItemRef/useItemRef';
import { useTimerRef } from 'shared/hooks/useTimerRef/useTimerRef';
import { Calendar, CalendarProps } from 'shared/ui/Calendar/Calendar';
import { InputProps } from 'shared/ui/Input/Input';
import { CLOSE_ANIMATION_TIME, DEFAULT_DATE_FORMAT, InputDate } from 'shared/ui/InputDate/InputDate';

import styles from 'shared/ui/InputDateSingle/InputDateSingle.css';

export interface InputDateSingleProps extends InputProps, Pick<CalendarProps, 'backYearStep' | 'nextYearStep'> {
    className?: string;

    date?: Nullable<Date>;
    minBookingDate?: Date;
    maxBookingDate?: Date;
    dateFormat?: Intl.DateTimeFormatOptions;

    onDateChange?(date: Optional<Date>): void;

    setRef?: ((element: Nullable<HTMLElement>) => void) | React.MutableRefObject<Nullable<HTMLElement>>;
}

const cx = cn.bind(styles);

export const InputDateSingle: React.FC<InputDateSingleProps> = function InputDateSingle({
    className,
    date,
    minBookingDate,
    maxBookingDate,
    backYearStep,
    nextYearStep,
    dateFormat = DEFAULT_DATE_FORMAT,
    onClick,
    onClear,
    onDateChange,
    setRef,
    ...otherProps
}) {
    const locale = getI18nLocale();
    const { ref: containerRef, itemRef } = useItemRef<HTMLElement>(setRef);

    const calendarRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const [currentDate, setCurrentDate] = useState<Nullable<Date>>(null);
    const [isFocused, setFocused] = useState<boolean>(false);
    const [activeMonths, setActiveMonths] = useState<Nullable<MonthType[]>>(null);
    const [isCalendarClosing, setCalendarClosing] = React.useState<boolean>(false);
    const [isCalendarVisible, setCalendarVisible] = React.useState<boolean>(false);

    const { style, update } = useFloatingUI(containerRef, calendarRef, isCalendarVisible, {
        placement: 'bottom-start',
        middleware: [flip({ crossAxis: false, padding: 8 })],
    });
    const timerRef = useTimerRef(0);

    React.useEffect(() => {
        setCurrentDate(date || null);
    }, [date]);

    const onCalendarCloseHandler = React.useCallback(() => {
        if (!isCalendarClosing) {
            setCalendarClosing(true);
            setFocused(false);

            timerRef.current = setTimeout(() => {
                setCalendarClosing(false);
                setCalendarVisible(false);
            }, CLOSE_ANIMATION_TIME) as unknown as number;
        }
    }, [timerRef, isCalendarClosing]);

    const onOutsideClickHandler = React.useCallback(
        ({ target }: MouseEvent) => {
            if (!containerRef.current?.contains(target as Node)) {
                onCalendarCloseHandler();
            }
        },
        [containerRef, onCalendarCloseHandler],
    );

    const onDatesChangeHandler = React.useCallback(
        ({ startDate }: OnDatesChangeProps) => {
            setCurrentDate(startDate);
            setActiveMonths(getCalendarMonths(startDate));

            if (startDate) {
                onCalendarCloseHandler();
            }

            if (onDateChange) {
                onDateChange(startDate ? startDate : undefined);
            }
        },
        [onDateChange, onCalendarCloseHandler],
    );

    const onInputClickHandler = React.useCallback(
        (event: React.MouseEvent<HTMLInputElement>) => {
            setFocused(true);
            setActiveMonths(getCalendarMonths(currentDate));
            setCalendarVisible(true);
            update();

            if (onClick) {
                onClick(event);
            }
        },
        [onClick, update, currentDate],
    );

    const onInputClearHandler = React.useCallback(
        (event: React.MouseEvent<HTMLSpanElement>) => {
            setFocused(true);
            setCalendarVisible(true);

            onDatesChangeHandler({ startDate: null } as OnDatesChangeProps);

            if (onClear) {
                onClear(event);
            }
        },
        [onClear, onDatesChangeHandler],
    );

    const onDayPickHandler = React.useCallback(
        (date: Date) => {
            onDatesChangeHandler({ startDate: date } as OnDatesChangeProps);
        },
        [onDatesChangeHandler],
    );

    const onPreviousClickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date));
    }, []);

    const onNextClickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date));
    }, []);

    const onYearPickHandler = React.useCallback((date: Date) => {
        setActiveMonths(getCalendarMonths(date));
    }, []);

    useClickOutside(calendarRef, onOutsideClickHandler);

    const dayContext = useDatepicker({
        startDate: currentDate,
        endDate: currentDate,
        minBookingDate,
        maxBookingDate,
        focusedInput: START_DATE,
        onDatesChange: onDatesChangeHandler,
    });

    return (
        <div
            className={cn(styles.inputDate, className)}
            ref={itemRef}
        >
            <InputDate
                {...otherProps}
                className={styles.inputInput}
                value={getDateString(currentDate, locale, dateFormat) || ''}
                focused={isFocused}
                onClick={onInputClickHandler}
                onClear={onInputClearHandler}
            />

            {isCalendarVisible && (
                <Calendar
                    className={cx(styles.calendar, { close: isCalendarClosing })}
                    style={style}
                    activeMonths={activeMonths}
                    dayContext={dayContext}
                    backYearStep={backYearStep}
                    nextYearStep={nextYearStep}
                    onPreviousClick={onPreviousClickHandler}
                    onNextClick={onNextClickHandler}
                    onYearPick={onYearPickHandler}
                    onDayPick={onDayPickHandler}
                    ref={calendarRef}
                />
            )}
        </div>
    );
};
