import * as React from 'react';
import cn from 'classnames/bind';

import { MenuType } from 'shared/consts/MenuType';

import CheckIcon from 'shared/ui/Icons/images/check-24.inline.svg';

import styles from 'shared/ui/MenuItem/MenuItem.css';

export type MenuItemAttributes = (JSX.IntrinsicElements['a'] & JSX.IntrinsicElements['span']) & {
    'data-navigate'?: string;
    'data-original-href'?: string;
    'data-trigger'?: string;
};

export interface MenuItemOptions<T extends string = string> extends MenuItemAttributes {
    // label vs content
    value: T;
    label?: string;
    disabled?: boolean;
    content?: React.ReactNode;
}

export interface MenuItemProps<T extends string = string> extends MenuItemOptions<T> {
    className?: string;
    menuType?: MenuType;
    visibleCheck?: boolean;
    checked?: boolean;
    selected?: boolean;

    onItemClick?(event: React.MouseEvent<HTMLLIElement>): void;
}

const cx = cn.bind(styles);

export function MenuItem<T extends string = string>({
    className,
    menuType,
    checked,
    selected,
    label,
    value,
    href,
    disabled,
    visibleCheck = true,
    content,
    onItemClick,
    ...otherProps
}: MenuItemProps<T>) {
    const TagName: string = href && !disabled ? 'a' : 'span';

    const attrs: MenuItemAttributes = {
        ...otherProps,
        className: styles.option,
        href,
    };

    return (
        <li
            className={cx(
                styles.menuItem,
                {
                    checked,
                    checkedColor: checked && !visibleCheck,
                    selected,
                    disabled,
                },
                [className],
            )}
            onClick={onItemClick}
            data-value={value}
            aria-selected={selected}
            role="option"
        >
            <TagName {...attrs}>
                {content || (
                    <>
                        {menuType && visibleCheck && <CheckIcon className={styles.checkIcon} />}

                        <span className={styles.label}>{label}</span>
                    </>
                )}
            </TagName>
        </li>
    );
}
