import * as React from 'react';
import cn from 'classnames/bind';

import styles from 'shared/ui/RadioButton/RadioButton.css';

export interface RadioButtonItemOptions<T extends string = string> {
    label: string;
    id: T;
    disabled?: boolean;
}

export interface RadioButtonProps<T extends string = string> {
    className?: string;

    items: RadioButtonItemOptions<T>[];
    value: T;
    disabled?: boolean;

    onChange?(value: T): void;
}

const cx = cn.bind(styles);

export const RadioButton = React.memo(function RadioButton({ className, items, value, disabled, onChange }) {
    const [currentValue, setCurrentValue] = React.useState<string>(value);

    React.useEffect(() => {
        setCurrentValue(value);
    }, [value]);

    const onChangeHandler = React.useCallback(
        ({ currentTarget }: React.ChangeEvent<HTMLInputElement>) => {
            if (onChange) {
                onChange(currentTarget.value as any);
            }
        },
        [onChange],
    );

    if (!items || !items.length) {
        return null;
    }

    return (
        <ul className={cx(styles.group, { disabled }, [className])}>
            {items.map(({ label, id, disabled: disabledItem }) => {
                const stringId = id.toString();
                const isDisabled = disabledItem || disabled;

                return (
                    <li
                        className={styles.item}
                        key={stringId}
                    >
                        <label className={cx(styles.radioButton, { disabled: isDisabled })}>
                            <input
                                className={styles.input}
                                type="radio"
                                value={stringId}
                                checked={currentValue === stringId}
                                disabled={isDisabled}
                                onChange={onChangeHandler}
                            />

                            <span className={styles.label}>{label}</span>
                        </label>
                    </li>
                );
            })}
        </ul>
    );
}) as <T extends string = string>(props: RadioButtonProps<T>) => React.ReactElement<RadioButtonProps<T>> | null;
