import * as React from 'react';
import cn from 'classnames/bind';

import { ScoringSize } from 'shared/consts/ScoringSize';
import { formatNumber } from 'shared/helpers/formatNumber/formatNumber';

import CrossIcon from 'shared/ui/Icons/images/cross-7.inline.svg';

import styles from 'shared/ui/Scoring/Scoring.css';

export interface ScoringProps {
    className?: string;
    size?: ScoringSize;
    value: Nullish<number>;
}

const cx = cn.bind(styles);

const SIZE: Record<ScoringSize, number> = {
    [ScoringSize.M]: 14,
    [ScoringSize.L]: 24,
};
const STROKE_WIDTH: Record<ScoringSize, number> = {
    [ScoringSize.M]: 1.5,
    [ScoringSize.L]: 2.5,
};

const PROGRESS_MULTIPLY = 10;
const CIRCLE_ANGLE = 180;
const REDUCTION = 0.3;

const RED_LIMIT = 3;
const YELLOW_LIMIT = 6;

export const Scoring: React.FC<ScoringProps> = React.memo(function Scoring({ className, size = ScoringSize.M, value }) {
    let svgIcon: Optional<React.ReactNode>;

    const rank = value || 0;
    const progress = Math.round(rank * PROGRESS_MULTIPLY);
    const center = SIZE[size] / 2;
    const rotate = CIRCLE_ANGLE / 2 + CIRCLE_ANGLE * REDUCTION;
    const r = center - STROKE_WIDTH[size] / 2;
    const circumference = Math.PI * r * 2;
    const offset = (circumference * (100 - progress * (1 - REDUCTION))) / 100;

    const circleArgs: React.SVGAttributes<SVGCircleElement> = {
        transform: `rotate(${rotate} ${center} ${center})`,
        cx: center,
        cy: center,
        r,
        strokeWidth: STROKE_WIDTH[size],
        strokeDasharray: circumference,
    };

    svgIcon = (
        <span className={styles.icon}>
            <svg
                viewBox={`0 0 ${SIZE[size]} ${SIZE[size]}`}
                xmlns="http://www.w3.org/2000/svg"
                fill="none"
            >
                <circle
                    {...circleArgs}
                    className={styles.circle}
                    strokeDashoffset={circumference * REDUCTION}
                />

                <circle
                    {...circleArgs}
                    className={cx(styles.circle, {
                        green: rank > YELLOW_LIMIT,
                        yellow: rank > RED_LIMIT && rank <= YELLOW_LIMIT,
                        red: rank <= RED_LIMIT,
                    })}
                    strokeDashoffset={offset}
                />
            </svg>

            {!value && <CrossIcon className={styles.cross} />}
        </span>
    );

    return (
        <span className={cx(styles.scoring, [size, className])}>
            {svgIcon}

            {value &&
                formatNumber(rank, {
                    minimumFractionDigits: 1,
                    maximumFractionDigits: 1,
                })}
        </span>
    );
});
