import * as React from 'react';
import cn from 'classnames/bind';

import { ButtonColor } from 'shared/consts/ButtonColor';
import { ButtonSize } from 'shared/consts/ButtonSize';
import { MenuType } from 'shared/consts/MenuType';
import { Dropdown, DropdownProps } from 'shared/ui/Dropdown/Dropdown';

import styles from 'shared/ui/Select/Select.css';

export interface SelectProps extends DropdownProps {
    className?: string;
    placeholder?: string;
    multiple?: boolean;
    withBackground?: boolean;

    onSelectChange?(value: string | string[]): void;
}

const cx = cn.bind(styles);

export const Select: React.FC<SelectProps> = function Select({
    className,
    items,
    placeholder = '',
    checkedMenu,
    multiple,
    equalWidth = true,
    disabled,
    withBackground,
    onSelectChange,
    onMenuChange,
    ...otherProps
}) {
    const [currentValue, setCurrentValue] = React.useState<Optional<string | string[]>>(checkedMenu);

    const onSelectChangeHandler = React.useCallback(
        (value: string | string[]) => {
            setCurrentValue(value);

            if (onSelectChange) {
                onSelectChange(value);
            }
        },
        [onSelectChange],
    );

    const currentOption = React.useMemo(() => {
        if (!currentValue || !currentValue.length) {
            return <span className={styles.placeholder}>{placeholder}</span>;
        }

        if (Array.isArray(currentValue) && currentValue.length > 1) {
            let labels: string[] = [];

            items.map(({ label, value }) => {
                if (currentValue.includes(value) && label) {
                    labels.push(label);
                }
            });

            return labels.join(', ');
        }

        const checkedOption = Array.isArray(currentValue) ? currentValue[0] : currentValue;

        return items.find(({ value }) => value === checkedOption)?.label;
    }, [items, placeholder, currentValue]);

    return (
        <Dropdown
            {...otherProps}
            className={cx(styles.select, { disabled, background: withBackground }, [className])}
            color={ButtonColor.SECONDARY}
            size={ButtonSize.M}
            label={currentOption}
            items={items}
            menuType={multiple ? MenuType.CHECK : MenuType.RADIO}
            checkedMenu={checkedMenu}
            equalWidth={equalWidth}
            disabled={disabled}
            onMenuChange={onSelectChangeHandler}
        />
    );
};
