import * as React from 'react';
import { Placement } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { MenuType } from 'shared/consts/MenuType';
import { useFloatingMenu } from 'shared/hooks/useFloatingMenu/useFloatingMenu';
import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { InputArrow, InputArrowProps } from 'shared/ui/InputArrow/InputArrow';
import { Menu, MenuProps } from 'shared/ui/Menu/Menu';

import styles from 'shared/ui/SelectEditable/SelectEditable.css';

export interface SelectEditableProps
    extends InputArrowProps,
        Pick<MenuProps, 'items' | 'menuType' | 'visibleCheck' | 'selected'> {
    checkedMenu: string;
    position?: Placement;
    equalWidth?: boolean;

    onMenuChange?(value: string): void;

    onMenuClick?(event: React.MouseEvent<HTMLLIElement>): void;
}

const cx = cn.bind(styles);

export const SelectEditable: React.FC<SelectEditableProps> = function SelectEditable({
    value,
    items,
    menuType = MenuType.RADIO,
    visibleCheck = false,
    selected,
    checkedMenu,
    position = 'bottom-start',
    equalWidth = true,
    onClick,
    onFocus,
    onMenuChange,
    onMenuClick,
    ...otherProps
}) {
    const Portal = usePortal();

    const referenceRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
    const floatingRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const [currentValue, setCurrentValue] = React.useState<Optional<string | string[]>>(checkedMenu);

    const { style, isMenuClosing, isMenuVisible, checked, onReferenceFocusHandler, onMenuItemClickHandler } =
        useFloatingMenu<HTMLInputElement>({
            referenceRef,
            floatingRef,
            value: checkedMenu,
            menuType,
            options: { placement: position, equalWidth },
            onReferenceClick: onClick,
            onReferenceFocus: onFocus,
            onMenuChange,
            onMenuClick,
        });

    React.useEffect(() => {
        setCurrentValue(checked);
    }, [checked]);

    const currentOption = React.useMemo(() => {
        if (!currentValue) {
            return;
        }

        return items.find(({ value }) => value === currentValue)?.label;
    }, [items, currentValue]);

    return (
        <div ref={referenceRef}>
            <InputArrow
                {...otherProps}
                value={currentOption || value}
                opened={isMenuVisible}
                onFocus={onReferenceFocusHandler}
            />

            {isMenuVisible && items && (
                <Portal>
                    <Menu
                        className={cx(styles.menu, { close: isMenuClosing })}
                        shadow
                        style={style}
                        items={items}
                        menuType={menuType}
                        visibleCheck={visibleCheck}
                        checked={checked}
                        selected={selected}
                        onItemClick={onMenuItemClickHandler}
                        ref={floatingRef}
                    />
                </Portal>
            )}
        </div>
    );
};
