import * as React from 'react';
import { useState } from 'react';
import { Placement } from '@floating-ui/react-dom';
import cn from 'classnames/bind';

import { MenuType } from 'shared/consts/MenuType';
import { FetchError, FetchRequestResult } from 'shared/helpers/fetchRequest/fetchRequest';
import { isEqual } from 'shared/helpers/isEqual/isEqual';
import {
    CacheRequestContext,
    useCacheRequestContext,
} from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { useFloatingMenu } from 'shared/hooks/useFloatingMenu/useFloatingMenu';
import { usePortal } from 'shared/hooks/usePortal/usePortal';
import { InputProps } from 'shared/ui/Input/Input';
import { InputArrow, InputArrowProps } from 'shared/ui/InputArrow/InputArrow';
import { Menu } from 'shared/ui/Menu/Menu';
import { MenuItemOptions } from 'shared/ui/MenuItem/MenuItem';

import styles from 'shared/ui/SelectEditable/SelectEditable.css';

export interface SuggestItem<T> {
    data: T;
    id: string;
    text: string;
    content?: React.ReactNode;
}

export interface SuggestProps<T> extends Omit<InputArrowProps, 'value' | 'opened' | 'onInputChange'> {
    inputComponent?: React.FC<InputProps>;
    position?: Placement;
    equalWidth?: boolean;

    initialValue?: SuggestValueState<T>;

    maxHeight?: number;

    dataProvider(textValue: string, cache?: CacheRequestContext): FetchRequestResult<T[]>;

    menuItemsProvider(items: T[]): SuggestItem<T>[];

    onSuggestChange?(textValue: string, dataValue: Optional<T>): void;
}

interface SuggestDataState<T> {
    data?: Optional<T[]>;
    error?: Optional<FetchError | Error>;
    isLoading: boolean;
}

export interface SuggestValueState<T> {
    text: string;
    data: Optional<T>;
    id: Optional<string>;
}

const cx = cn.bind(styles);

export function Suggest<T>({
    inputComponent: InputComponent,
    position = 'bottom-start',
    equalWidth = true,
    icon,
    hasArrow,
    initialValue,
    dataProvider,
    menuItemsProvider,
    maxHeight,
    onSuggestChange,
    onBlur,
    onClick,
    onFocus,
    ...otherProps
}: SuggestProps<T>) {
    const cacheContext = useCacheRequestContext();

    const Portal = usePortal();

    const [dataState, setDataState] = React.useState<SuggestDataState<T>>({ isLoading: false });
    const [value, setValue] = React.useState<SuggestValueState<T>>(
        initialValue || { text: '', data: undefined, id: undefined },
    );
    const [, setVer] = useState<number>(0);

    const referenceRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;
    const floatingRef = React.useRef() as React.MutableRefObject<Nullable<HTMLDivElement>>;

    const suggestItems = React.useMemo<SuggestItem<T>[]>(() => {
        if (dataState.data) {
            return menuItemsProvider(dataState.data);
        }

        return [];
    }, [dataState, menuItemsProvider]);

    const menuItems = React.useMemo<MenuItemOptions[]>(() => {
        return suggestItems.map(({ id, text, content }) => {
            return {
                value: id,
                label: text,
                content,
            };
        });
    }, [suggestItems]);

    const setFloatingRef = React.useCallback((ref: Nullable<HTMLDivElement>) => {
        if (floatingRef.current !== ref) {
            floatingRef.current = ref;

            setVer((ver) => ver + 1);
        }
    }, []);

    const onMenuChange = React.useCallback(
        (checked: string | string[]) => {
            let checkedId: Optional<string> = undefined;

            if (typeof checked === 'string') {
                checkedId = checked;
            }

            if (checkedId === value.id) {
                return;
            }

            let nextValueText: string = value.text;
            let nextValueData: Optional<T> = undefined;

            if (checkedId) {
                let suggestItem = suggestItems.find((item) => checkedId === item.id);

                if (suggestItem) {
                    nextValueData = suggestItem.data;
                    nextValueText = suggestItem.text;
                }
            }

            const nextValue = { text: nextValueText, data: nextValueData, id: checkedId };

            if (!isEqual(value, nextValue)) {
                setValue(nextValue);

                if (onSuggestChange) {
                    onSuggestChange(nextValueText, nextValueData);
                }
            }
        },
        [value, setValue, suggestItems, onSuggestChange],
    );

    const onInputChange = React.useCallback(
        (text: string) => {
            setValue({ text, data: undefined, id: undefined });

            if (onSuggestChange) {
                onSuggestChange(text, undefined);
            }
        },
        [onSuggestChange],
    );

    const {
        style,
        isMenuClosing,
        isMenuVisible,
        checked,
        onReferenceFocusHandler,
        onReferenceBlurHandler,
        onMenuItemClickHandler,
    } = useFloatingMenu<HTMLInputElement>({
        referenceRef,
        floatingRef,
        value: value.id,
        menuType: MenuType.RADIO,
        options: { placement: position, equalWidth },
        onReferenceClick: onClick,
        onReferenceFocus: onFocus,
        onReferenceBlur: onBlur,
        onMenuChange,
    });

    React.useEffect(() => {
        setDataState((state) => ({ ...state, isLoading: true }));

        const request = dataProvider(value.text, cacheContext);

        request.ready().then(
            (data) => setDataState(() => ({ data, isLoading: false })),
            (error) => {
                if (error instanceof FetchError && error.canceled) {
                    return;
                }

                setDataState(() => ({ error, isLoading: false }));
            },
        );

        return request.cancel;
    }, [dataProvider, value.text, cacheContext]);

    const hasMenu = isMenuVisible && suggestItems && Boolean(suggestItems.length);

    let input;

    if (InputComponent) {
        input = (
            <InputComponent
                {...otherProps}
                value={value.text}
                onFocus={onReferenceFocusHandler}
                onBlur={onReferenceBlurHandler}
                onInputChange={onInputChange}
            />
        );
    } else {
        input = (
            <InputArrow
                {...otherProps}
                icon={icon}
                opened={hasMenu}
                hasArrow={hasArrow}
                value={value.text}
                onFocus={onReferenceFocusHandler}
                onBlur={onReferenceBlurHandler}
                onInputChange={onInputChange}
            />
        );
    }

    return (
        <div ref={referenceRef}>
            {input}

            {isMenuVisible && (
                <Portal>
                    <Menu
                        className={cx(styles.menu, { close: isMenuClosing })}
                        shadow
                        style={{ ...style, maxHeight }}
                        items={menuItems}
                        menuType={MenuType.RADIO}
                        visibleCheck={false}
                        checked={checked}
                        // selected={selected}
                        onItemClick={onMenuItemClickHandler}
                        ref={setFloatingRef}
                    />
                </Portal>
            )}
        </div>
    );
}
