import * as React from 'react';
import cn from 'classnames/bind';

import styles from 'shared/ui/Table/Table.css';

export interface TableComponentProps {
    className?: string;
}

export interface TableCellComponentProps {
    gap?: boolean;
}

export type TableOptions = JSX.IntrinsicElements['table'];

export interface TableProps extends TableComponentProps, TableOptions {
    hasHover?: boolean;
}

export interface TableComponents {
    Colgroup: React.FC<TableComponentProps & JSX.IntrinsicElements['colgroup']>;
    Col: React.FC<TableCellComponentProps & TableComponentProps & JSX.IntrinsicElements['col']>;
    Head: React.FC<TableComponentProps & JSX.IntrinsicElements['thead']>;
    Body: React.FC<TableComponentProps & JSX.IntrinsicElements['tbody']>;
    Row: React.FC<TableComponentProps & JSX.IntrinsicElements['tr']>;
    Th: React.FC<TableCellComponentProps & TableComponentProps & JSX.IntrinsicElements['th']>;
    Td: React.FC<TableCellComponentProps & TableComponentProps & JSX.IntrinsicElements['td']>;
}

const cx = cn.bind(styles);

const Table: React.FC<TableProps> & TableComponents = function Table({
    className,
    hasHover = true,
    children,
    ...args
}) {
    return (
        <table
            className={cx(styles.table, { hover: hasHover }, [className])}
            {...args}
        >
            {children}
        </table>
    );
};

Table.Colgroup = function TableColgroup({ children, ...args }) {
    return <colgroup {...args}>{children}</colgroup>;
};

Table.Col = function TableCol({ className, gap, children, ...args }) {
    return (
        <col
            className={cx(styles.col, { gap }, [className])}
            {...args}
        >
            {children}
        </col>
    );
};

Table.Head = function TableHead({ children, ...args }) {
    return <thead {...args}>{children}</thead>;
};

Table.Body = function TableBody({ children, ...args }) {
    return <tbody {...args}>{children}</tbody>;
};

Table.Row = function TableRow({ className, children, ...args }) {
    return (
        <tr
            className={cn(styles.tr, className)}
            {...args}
        >
            {children}
        </tr>
    );
};

Table.Th = function TableTh({ className, gap, children, ...args }) {
    return (
        <th
            className={cx(styles.th, { gap }, [className])}
            {...args}
        >
            {children}
        </th>
    );
};

Table.Td = function TableTd({ className, gap, children, ...args }) {
    return (
        <td
            className={cx(styles.td, { gap }, [className])}
            {...args}
        >
            {children}
        </td>
    );
};

export { Table };
