import * as React from 'react';
import cn from 'classnames/bind';

import { TextareaSize } from 'shared/consts/TextareaSize';
import { useItemRef } from 'shared/hooks/useItemRef/useItemRef';
import { InputIcon } from 'shared/ui/InputIcon/InputIcon';

import CloseIcon16 from 'shared/ui/Icons/images/close-16.inline.svg';

import styles from 'shared/ui/Textarea/Textarea.css';

export type TextareaAttributes = JSX.IntrinsicElements['textarea'];

export interface TextareaProps extends TextareaAttributes {
    className?: string;
    textareaSize?: TextareaSize;
    expandable?: boolean;

    disabled?: boolean;

    hasError?: boolean;
    hasClear?: boolean;

    hasAutoFocus?: boolean;
    shouldFocusOnClear?: boolean;

    onClear?(event: React.MouseEvent<HTMLSpanElement>): void;

    onTextareaChange?(value: string): void;

    setRef?: ((element: Nullable<HTMLElement>) => void) | React.MutableRefObject<Nullable<HTMLElement>>;
}

const cx = cn.bind(styles);

const DEFAULT_ROWS = 2;

function setHeight(target: HTMLElement) {
    target.style.height = '';
    target.style.height = `${Math.max(0, target.scrollHeight + target.offsetHeight - target.clientHeight)}px`;
}

export const Textarea: React.FC<TextareaProps> = function Textarea({
    className,
    rows = DEFAULT_ROWS,
    textareaSize,
    expandable,
    placeholder = '',
    value,
    disabled,
    hasError,
    hasClear,
    hasAutoFocus,
    shouldFocusOnClear = true,
    onChange,
    onClear,
    onTextareaChange,
    setRef,
    children,
    ...otherProps
}) {
    const { ref, itemRef } = useItemRef<HTMLElement>(setRef);

    React.useEffect(() => {
        if (expandable && ref && ref.current) {
            setHeight(ref.current);
        }
    }, [expandable, value, ref.current]);

    React.useEffect(() => {
        if (hasAutoFocus && ref.current) {
            ref.current.focus();
        }
    }, [hasAutoFocus, ref.current]);

    const onClearHandler = React.useCallback(
        (event: React.MouseEvent<HTMLSpanElement>) => {
            if (onTextareaChange) {
                onTextareaChange('');
            }

            if (onClear) {
                onClear(event);
            }

            if (shouldFocusOnClear && ref.current) {
                ref.current.focus();
            }
        },
        [onClear, shouldFocusOnClear, ref.current],
    );

    const onChangeHandler = React.useCallback(
        (event: React.ChangeEvent<HTMLTextAreaElement>) => {
            if (onTextareaChange) {
                onTextareaChange(event.target.value);
            }

            if (onChange) {
                onChange(event);
            }
        },
        [onTextareaChange, onChange],
    );

    const isClearVisible = Boolean(!disabled && hasClear && value?.toString().length);

    const attrs: TextareaAttributes = {
        ...otherProps,
        className: cx(styles.textareaInput, { expandable }),
        rows,
        placeholder,
        value,
        disabled,
        onChange: onChangeHandler,
        ref: itemRef,
    };

    return (
        <div
            className={cx(
                styles.textarea,
                {
                    hasError,
                    hasClear,
                    disabled,
                },
                [textareaSize, className],
            )}
        >
            <textarea {...attrs} />

            {hasClear && (
                <InputIcon
                    className={styles.clear}
                    icon={CloseIcon16}
                    visible={isClearVisible}
                    onClick={onClearHandler}
                />
            )}

            {children}

            <span className={styles.border}></span>
        </div>
    );
};
