import * as React from 'react';
import cn from 'classnames/bind';

import styles from 'shared/ui/Toggle/Toggle.css';

export interface ToggleItemOptions {
    label: React.ReactNode;
    id: number | string;
    checked?: boolean;
    disabled?: boolean;
}

export interface ToggleProps extends ToggleItemOptions {
    className?: string;

    onChange?(checked: boolean, value: string): void;
}

const cx = cn.bind(styles);

export const Toggle = React.forwardRef<HTMLLabelElement, ToggleProps>(function Toggle(
    { className, label, id, checked, disabled, onChange },
    ref,
) {
    const onChangeHandler = React.useCallback(
        ({ currentTarget }: React.ChangeEvent<HTMLInputElement>) => {
            if (onChange) {
                onChange(currentTarget.checked, currentTarget.value);
            }
        },
        [onChange],
    );

    return (
        <label
            className={cx(styles.toggle, { disabled }, [className])}
            ref={ref}
        >
            <input
                className={styles.input}
                type="checkbox"
                value={id.toString()}
                checked={checked}
                disabled={disabled}
                onChange={onChangeHandler}
            />

            <span
                className={styles.display}
                aria-checked={checked}
                hidden
                role="checkbox"
            />

            {label && <span className={styles.label}>{label}</span>}
        </label>
    );
});
