import React, { useState } from 'react';

import { ClearFilters } from 'features/ClearFilters';
import { FilterDate } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { CarSessionsFilter } from 'entities/Car/consts/CarSessionsFilter';
import { EXISTING_CAR_SESSIONS_FILTERS, INITIAL_CAR_SESSIONS_FILTERS } from 'entities/Car/consts/filters';

import { DateFilter } from 'shared/consts/DateFilter';
import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { CarSessionsFiltersOptions } from 'components/Cars/CarCard/CarSessionsTable';

import { i18n } from 'widgets/CarSessionsFilters/ui/CarSessionsFilters/CarSessionsFilters.i18n';

export interface CarSessionsFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange(filters: CarSessionsFiltersOptions): void;
}

export const CarSessionsFilters: React.FC<CarSessionsFiltersProps> = function CarSessionsFilters({
    className,
    offsetTop,
    onFiltersChange,
}) {
    const [filters, setFilters] = useState<CarSessionsFiltersOptions>(INITIAL_CAR_SESSIONS_FILTERS);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Record<DateFilter, Nullable<number>>) => {
            let filtersObj = { ...filters };

            if (type === CarSessionsFilter.DATE) {
                [DateFilter.SINCE, DateFilter.UNTIL].forEach((key) => {
                    filtersObj[key] = value?.[key];
                });
            }

            setFilters(filtersObj);
            onFiltersChange(filtersObj);
        },
        [filters, onFiltersChange],
    );

    const onClearFiltersHandler = React.useCallback(() => {
        setFilters(INITIAL_CAR_SESSIONS_FILTERS);
        onFiltersChange(INITIAL_CAR_SESSIONS_FILTERS);
    }, [onFiltersChange]);

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={
                <ClearFilters
                    filters={EXISTING_CAR_SESSIONS_FILTERS}
                    onClick={onClearFiltersHandler}
                />
            }
            isClearVisible={hasActiveFilters<CarSessionsFiltersOptions>(filters)}
        >
            <FilterDate
                title={i18n('Ride date')}
                type={CarSessionsFilter.DATE}
                filters={filters}
                onChangeFilter={onChangeFilterHandler}
            />
        </FiltersContainer>
    );
};
