import * as React from 'react';
import { useLocation } from 'react-router-dom';
import cn from 'classnames';

import { getHideTaxiParkDataFlag } from 'utils/getHideTaxiParkDataFlag';
import { isCarAddFlag } from 'utils/isCarAddFlag';
import { isCarsGroupsAccess } from 'utils/isCarsGroupsAccess';
import { isShowCarService } from 'utils/isShowCarService';

import { CarCreate } from 'features/CarCreate';
import { CreateGroup } from 'features/CreateGroup';
import { CustomizeTable } from 'features/CustomizeTable/ui/CustomizeTable/CustomizeTable';
import { DownloadSection, DownloadXLS } from 'features/DownloadXLS';
import { SaveReport } from 'features/SaveReport';

import { CarsFilter } from 'entities/Car/consts/CarsFilter';
import { filterCarsTableColumns } from 'entities/Car/helpers/filterCarsTableColumns/filterCarsTableColumns';
import { ReportSection } from 'entities/Report/consts/ReportSection';

import { ORDER_DESC_CGI, ORDER_FIELD_CGI } from 'shared/consts/order';
import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';

import { CarsFiltersOptions } from 'components/Cars';
import { CARS_HEADER, DEFAULT_CARS_COLUMNS, REQUIRED_CARS_COLUMNS } from 'components/Cars/CarsTable/constants';

import styles from 'widgets/CarsContentHeader/ui/CarsContentHeader/CarsContentHeader.css';

export interface CarsContentHeaderProps {
    className?: string;
    tagsFilter?: string;
    filters: CarsFiltersOptions;

    // @todo: fix types
    getData(options?: any): Promise<any>;

    abortGettingCars(): void;

    onCarsTableColumnsChange(columns: string[]): void;
}

export const CarsContentHeader: React.FC<CarsContentHeaderProps> = function CarsContentHeader({
    className,
    tagsFilter,
    filters,
    getData,
    abortGettingCars,
    onCarsTableColumnsChange,
}) {
    const location = useLocation();

    const hasFilters = hasActiveFilters<CarsFiltersOptions>(filters);

    // @todo: check `filters` and compare with `filterValues`. Perhaps we should use `filters`.
    const filterValues = React.useMemo(() => {
        const searchParams = new URLSearchParams(location?.search);

        const carClass = searchParams.get(CarsFilter.CLASS);

        const parkId = searchParams.get(CarsFilter.PARK_ID);

        const orderField = searchParams.get(ORDER_FIELD_CGI);
        const orderDesc = searchParams.get(ORDER_DESC_CGI);

        return {
            // Place for custom filter
            ...(carClass ? { class: carClass } : {}),

            telematics: filters.telematics,
            camera: filters.camera,

            ...(!getHideTaxiParkDataFlag()
                ? {
                      no_sh: filters.no_sh,
                      fresh_issue_date: filters.fresh_issue_date,
                  }
                : {}),
            ...(parkId ? { park_id: parkId } : {}),
            ...(isShowCarService() ? { service: filters.service } : {}),
            status: filters.status,
            signalq_status: filters.signalq_status,
            ...(orderField
                ? {
                      order_field: orderField,
                      order_desc: orderDesc,
                  }
                : {}),
        };
    }, [location, filters]);

    const onCustomizeModalCloseHandler = React.useCallback((columns?: string[]) => {
        if (columns) {
            onCarsTableColumnsChange(columns);
        }
    }, []);

    const filterColumns = React.useCallback((arr: string[]) => filterCarsTableColumns<string>(arr), []);

    // @todo: add enum tableType
    return (
        <div className={cn(styles.carsContentHeader, className)}>
            {isCarAddFlag() && <CarCreate />}

            <DownloadXLS
                section={DownloadSection.CARS}
                getData={getData}
                abortGettingData={abortGettingCars}
            />

            {hasFilters && (
                <>
                    <SaveReport
                        section={ReportSection.CARS}
                        filters={filterValues as CarsFiltersOptions}
                    />

                    <CreateGroup tagsFilter={isCarsGroupsAccess() ? tagsFilter : null} />
                </>
            )}

            <CustomizeTable
                className={styles.customizeTable}
                tableType="cars"
                headers={CARS_HEADER}
                defaultColumns={DEFAULT_CARS_COLUMNS}
                requiredColumns={REQUIRED_CARS_COLUMNS}
                filterColumns={filterColumns}
                onCloseModal={onCustomizeModalCloseHandler}
            />
        </div>
    );
};
