import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { getHideTaxiParkDataFlag } from 'utils/getHideTaxiParkDataFlag';
import { isShowCarCameraStatusFiltersFlag } from 'utils/isShowCarCameraStatusFiltersFlag';
import { isShowCarService } from 'utils/isShowCarService';
import { isShowCarStatusFilters } from 'utils/isShowCarStatusFilters';
import { isShowHardwareFilterFlag } from 'utils/isShowHardwareFilterFlag';

import { ClearFilters } from 'features/ClearFilters';

import {
    CarsCarLoveClassFilter,
    CarsFreshFilter,
    CarsNoShFilter,
    CarsParksFilter,
    CarsServiceFilter,
    CarsStatusFilter,
} from 'entities/Car';
import { CarsFilter } from 'entities/Car/consts/CarsFilter';
import {
    BOOLEAN_CARS_FILTERS,
    EXISTING_CARS_FILTERS,
    INITIAL_CARS_FILTERS,
    STRING_CARS_FILTERS,
} from 'entities/Car/consts/filters';
import { CarsCameraStatusFilter } from 'entities/Car/ui/CarsCameraStatusFilter/CarsCameraStatusFilter';
import { CarsHardwareFilter } from 'entities/Car/ui/CarsHardwareFilter/CarsHardwareFilter';

import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { setFiltersParams } from 'shared/helpers/setFiltersParams/setFiltersParams';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { CarsFiltersOptions } from 'components/Cars';

export interface CarsFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange?(filters: CarsFiltersOptions): void;
}

export const CarsFilters: React.FC<CarsFiltersProps> = function CarsFilters({ className, offsetTop, onFiltersChange }) {
    const location = useLocation();
    const history = useHistory();

    const showTaxiParkData = !getHideTaxiParkDataFlag();
    const showHardwareFilterFlag = isShowHardwareFilterFlag();
    const showCarService = isShowCarService();
    const showCarStatusFilters = isShowCarStatusFilters();
    const showCarCameraStatusFilters = isShowCarCameraStatusFiltersFlag();

    const filters = useSearchFilters<CarsFiltersOptions>(INITIAL_CARS_FILTERS, EXISTING_CARS_FILTERS, {
        stringFilters: STRING_CARS_FILTERS,
        booleanFilters: BOOLEAN_CARS_FILTERS,
    });

    const hardwareValues = React.useMemo(() => {
        return [
            filters[CarsFilter.CAMERA] ? CarsFilter.CAMERA : undefined,
            filters[CarsFilter.TELEMATICS] ? CarsFilter.TELEMATICS : undefined,
        ].filter(Boolean) as string[];
    }, [filters]);

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: string | boolean) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    const onChangeHardwareFilterHandler = React.useCallback(
        (value: Record<string, boolean>) => {
            const searchParams = new URLSearchParams(location?.search);

            Object.keys(value).forEach((name) => {
                setFiltersParams(searchParams, name, Boolean(value[name]));
            });

            history.push(`${location.pathname}?${searchParams.toString()}`);
        },
        [location],
    );

    const hasFilters = React.useMemo(() => {
        return hasActiveFilters<CarsFiltersOptions>(filters);
    }, [filters]);

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={<ClearFilters filters={EXISTING_CARS_FILTERS} />}
            isClearVisible={hasFilters}
        >
            <CarsCarLoveClassFilter
                value={filters[CarsFilter.CLASS]}
                onChangeFilter={onChangeFilterHandler}
            />

            {showHardwareFilterFlag && (
                <CarsHardwareFilter
                    values={hardwareValues}
                    onChangeFilter={onChangeHardwareFilterHandler}
                />
            )}

            {showTaxiParkData && (
                <>
                    <CarsNoShFilter
                        checked={Boolean(filters[CarsFilter.NO_SH])}
                        onChangeFilter={onChangeFilterHandler}
                    />

                    <CarsFreshFilter
                        value={filters[CarsFilter.FRESH_ISSUE_DATE]}
                        onChangeFilter={onChangeFilterHandler}
                    />

                    <CarsParksFilter
                        values={filters[CarsFilter.PARK_ID]}
                        onChangeFilter={onChangeFilterHandler}
                    />
                </>
            )}

            {showCarService && (
                <CarsServiceFilter
                    checked={Boolean(filters[CarsFilter.SERVICE])}
                    onChangeFilter={onChangeFilterHandler}
                />
            )}

            {showCarStatusFilters && (
                <CarsStatusFilter
                    values={filters[CarsFilter.STATUS]}
                    onChangeFilter={onChangeFilterHandler}
                />
            )}

            {showCarCameraStatusFilters && (
                <CarsCameraStatusFilter
                    values={filters[CarsFilter.SIGNALQ_STATUS]}
                    onChangeFilter={onChangeFilterHandler}
                />
            )}
        </FiltersContainer>
    );
};
