import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ClearFilters } from 'features/ClearFilters';

import { ManageSignalsPriorityFilter } from 'entities/Signal';
import {
    EXISTING_MANAGE_SIGNALS_FILTERS,
    INITIAL_MANAGE_SIGNALS_FILTERS,
    STRING_MANAGE_SIGNALS_FILTERS,
} from 'entities/Signal/consts/filters';
import { ManageSignalsFilter } from 'entities/Signal/consts/ManageSignalsFilter';

import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { ManageSignalsFiltersOptions } from 'components/Signals/ManageSignals';

export interface ManageSignalsFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange?(filters: ManageSignalsFiltersOptions): void;
}

export const ManageSignalsFilters: React.FC<ManageSignalsFiltersProps> = function ManageSignalsFilters({
    className,
    offsetTop,
    onFiltersChange,
}) {
    const location = useLocation();
    const history = useHistory();

    const filters = useSearchFilters<ManageSignalsFiltersOptions>(
        INITIAL_MANAGE_SIGNALS_FILTERS,
        EXISTING_MANAGE_SIGNALS_FILTERS,
        {
            stringFilters: STRING_MANAGE_SIGNALS_FILTERS,
        },
    );

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Optional<string>) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={<ClearFilters filters={EXISTING_MANAGE_SIGNALS_FILTERS} />}
            isClearVisible={hasActiveFilters<ManageSignalsFiltersOptions>(filters)}
        >
            <ManageSignalsPriorityFilter
                values={filters[ManageSignalsFilter.PRIORITY]}
                onChangeFilter={onChangeFilterHandler}
            />
        </FiltersContainer>
    );
};
