import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ClearFilters } from 'features/ClearFilters';

import { PortfolioPeriodFilter } from 'entities/Portfolio';
import {
    EXISTING_PORTFOLIO_FILTERS,
    INITIAL_PORTFOLIO_FILTERS,
    STRING_PORTFOLIO_FILTERS,
} from 'entities/Portfolio/consts/filters';
import { PortfolioFilter } from 'entities/Portfolio/consts/PortfolioFilter';

import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { PortfolioFiltersOptions } from 'components/Portfolio';

export interface PortfolioFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange?(filters: PortfolioFiltersOptions): void;
}

export const PortfolioFilters: React.FC<PortfolioFiltersProps> = function PortfolioFilters({
    className,
    offsetTop,
    onFiltersChange,
}) {
    const location = useLocation();
    const history = useHistory();

    const filters = useSearchFilters<PortfolioFiltersOptions>(INITIAL_PORTFOLIO_FILTERS, EXISTING_PORTFOLIO_FILTERS, {
        stringFilters: STRING_PORTFOLIO_FILTERS,
    });

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: string) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={<ClearFilters filters={EXISTING_PORTFOLIO_FILTERS} />}
            isClearVisible={hasActiveFilters<PortfolioFiltersOptions>(filters)}
        >
            <PortfolioPeriodFilter
                value={filters[PortfolioFilter.COMPARE_PERIOD]}
                onChangeFilter={onChangeFilterHandler}
            />
        </FiltersContainer>
    );
};
