import * as React from 'react';

import {
    RentalRetireeAgreementModal,
    RentalRetireeDocumentsModal,
    RentalRetireeFinishRideModal,
    RentalRetireePaymentModal,
    RentalRetireeStartRideModal,
} from 'features/RentalRetiree';

import { CarRentalRetireeStepSchema } from 'entities/Car';

export interface RentalRetireeProps {
    offerId: string;
    tagId: string;
    userId: string;
    initialStep: Optional<CarRentalRetireeStepSchema>;

    onChange?(): void;

    onClose?(): void;
}

// @todo: add storybook test
export const RentalRetiree: React.FC<RentalRetireeProps> = function RentalRetiree({
    offerId,
    tagId,
    userId,
    initialStep,
    onChange,
    onClose,
}) {
    const [step, setStep] = React.useState<Optional<CarRentalRetireeStepSchema>>(initialStep);

    const onFormSubmitHandler = React.useCallback((formStep: Optional<CarRentalRetireeStepSchema>) => {
        setStep(formStep);
    }, []);

    const onCloseHandler = React.useCallback(() => {
        setStep(undefined);

        if (onClose) {
            onClose();
        }
    }, [onClose]);

    const onChangeHandler = React.useCallback(() => {
        onCloseHandler();

        if (onChange) {
            onChange();
        }
    }, [onChange, onCloseHandler]);

    const onDocumentFormSubmitHandler = React.useCallback(() => {
        onFormSubmitHandler('agreement');
    }, [onFormSubmitHandler]);

    const onAgreementFormSubmitHandler = React.useCallback(() => {
        onFormSubmitHandler('payment');
    }, [onFormSubmitHandler]);

    const onPaymentFormSubmitHandler = React.useCallback(() => {
        onFormSubmitHandler('start');
    }, [onFormSubmitHandler]);

    return (
        <React.Suspense fallback={null}>
            {step === 'documents' && (
                <RentalRetireeDocumentsModal
                    userId={userId}
                    onFormSubmit={onDocumentFormSubmitHandler}
                    onClose={onCloseHandler}
                />
            )}

            {step === 'agreement' && (
                <RentalRetireeAgreementModal
                    offerId={offerId}
                    userId={userId}
                    onFormSubmit={onAgreementFormSubmitHandler}
                    onClose={onCloseHandler}
                />
            )}

            {step === 'payment' && (
                <RentalRetireePaymentModal
                    tagId={tagId}
                    offerId={offerId}
                    userId={userId}
                    onFormSubmit={onPaymentFormSubmitHandler}
                    onClose={onCloseHandler}
                />
            )}

            {step === 'start' && (
                <RentalRetireeStartRideModal
                    offerId={offerId}
                    userId={userId}
                    onFormSubmit={onChangeHandler}
                    onClose={onCloseHandler}
                />
            )}

            {step === 'finish' && (
                <RentalRetireeFinishRideModal
                    offerId={offerId}
                    userId={userId}
                    onFormSubmit={onChangeHandler}
                    onClose={onCloseHandler}
                />
            )}
        </React.Suspense>
    );
};
