import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ClearFilters } from 'features/ClearFilters';

import { ReportsPeriodFilter, ReportsSectionFilter } from 'entities/Report';
import {
    EXISTING_REPORTS_FILTERS,
    INITIAL_REPORTS_FILTERS,
    STRING_REPORTS_FILTERS,
} from 'entities/Report/consts/filters';
import { ReportsFilter } from 'entities/Report/consts/ReportsFilter';

import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { ReportsFiltersOptions } from 'components/Reports';

export interface ReportsFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange?(filters: ReportsFiltersOptions): void;
}

export const ReportsFilters: React.FC<ReportsFiltersProps> = function ReportsFilters({
    className,
    offsetTop,
    onFiltersChange,
}) {
    const location = useLocation();
    const history = useHistory();

    const filters = useSearchFilters<ReportsFiltersOptions>(INITIAL_REPORTS_FILTERS, EXISTING_REPORTS_FILTERS, {
        stringFilters: STRING_REPORTS_FILTERS,
    });

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = (type: string, value: any) => {
        const searchParams = new URLSearchParams(location?.search);

        history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
    };

    return (
        <>
            <FiltersContainer
                className={className}
                offsetTop={offsetTop}
                clearButton={<ClearFilters filters={EXISTING_REPORTS_FILTERS} />}
                isClearVisible={hasActiveFilters<ReportsFiltersOptions>(filters)}
            >
                <ReportsSectionFilter
                    values={filters[ReportsFilter.SECTION]}
                    onChangeFilter={onChangeFilterHandler}
                />

                <ReportsPeriodFilter
                    values={filters[ReportsFilter.PERIOD]}
                    onChangeFilter={onChangeFilterHandler}
                />
            </FiltersContainer>
        </>
    );
};
