import * as React from 'react';
import cn from 'classnames';

import { ONE_SECOND } from 'constants/constants';

import { CustomizeTable } from 'features/CustomizeTable/ui/CustomizeTable/CustomizeTable';
import { DownloadSection, DownloadXLS } from 'features/DownloadXLS';
import { SaveReport } from 'features/SaveReport';

import { ReportSection } from 'entities/Report/consts/ReportSection';
import { filterRidesTableColumns } from 'entities/Ride/helpers/filterRidesTableColumns/filterRidesTableColumns';

import { DateFilter } from 'shared/consts/DateFilter';
import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';

import { RidesFiltersOptions } from 'components/Rides';
import {
    getDefaultRidesColumn,
    getRidesTableHeaders,
    REQUIRED_RIDES_HEADERS,
} from 'components/Rides/RidesTable/constants';

import styles from 'widgets/RidesContentHeader/ui/RidesContentHeader/RidesContentHeader.css';

export interface RidesContentHeaderProps {
    className?: string;
    filters: RidesFiltersOptions;
    hasDriverColumn?: boolean;

    // @todo: fix types
    getData(options?: any): Promise<any>;

    onCarsTableColumnsChange(columns: string[]): void;
}

export const RidesContentHeader: React.FC<RidesContentHeaderProps> = function RidesContentHeader({
    className,
    filters,
    hasDriverColumn = true,
    getData,
    onCarsTableColumnsChange,
}) {
    const hasFilters = hasActiveFilters<RidesFiltersOptions>(filters);

    const getDataHandler = React.useCallback(
        (currentUntil?: number) => {
            const urlSearchParams = new URLSearchParams(location?.search);
            const since = urlSearchParams.get(DateFilter.SINCE) || null;
            const until = currentUntil || urlSearchParams.get(DateFilter.UNTIL) || Math.floor(Date.now() / ONE_SECOND);

            // @todo: check this line. Probably `userId` has never used on this page.
            const userId = urlSearchParams.get('userId');

            return getData({ since, until, numdoc: 200, userId });
        },
        [getData],
    );

    const onCustomizeModalCloseHandler = React.useCallback((columns?: string[]) => {
        if (columns) {
            onCarsTableColumnsChange(columns);
        }
    }, []);

    const filterColumns = React.useCallback((arr: string[]) => filterRidesTableColumns<string>(arr), []);

    // @todo: add enum tableType
    return (
        <div className={cn(styles.ridesContentHeader, className)}>
            <DownloadXLS
                section={DownloadSection.RIDES}
                getData={getDataHandler}
            />

            {hasFilters && (
                <SaveReport
                    section={ReportSection.RIDES}
                    filters={filters}
                />
            )}

            <CustomizeTable
                className={styles.customizeTable}
                tableType="rides"
                headers={getRidesTableHeaders(hasDriverColumn)}
                defaultColumns={getDefaultRidesColumn(hasDriverColumn)}
                requiredColumns={REQUIRED_RIDES_HEADERS}
                filterColumns={filterColumns}
                onCloseModal={onCustomizeModalCloseHandler}
            />
        </div>
    );
};
