import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { ClearFilters } from 'features/ClearFilters';
import { FilterDate } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { DATE_RIDES_FILTERS, EXISTING_RIDES_FILTERS, INITIAL_RIDES_FILTERS } from 'entities/Ride/consts/filters';
import { RidesFilter } from 'entities/Ride/consts/RidesFilter';

import { DateFilter } from 'shared/consts/DateFilter';
import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { RidesFiltersOptions } from 'components/Rides';

import { i18n } from 'widgets/RidesFilters/ui/RidesFilters/RidesFilters.i18n';

export interface RidesFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    onFiltersChange?(filters: RidesFiltersOptions): void;
}

export const RidesFilters: React.FC<RidesFiltersProps> = function RidesFilters({
    className,
    offsetTop,
    onFiltersChange,
}) {
    const location = useLocation();
    const history = useHistory();

    const filters = useSearchFilters<RidesFiltersOptions>(INITIAL_RIDES_FILTERS, EXISTING_RIDES_FILTERS, {
        dateFilters: DATE_RIDES_FILTERS,
    });

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Record<DateFilter, Nullable<number>>) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={<ClearFilters filters={EXISTING_RIDES_FILTERS} />}
            isClearVisible={hasActiveFilters<RidesFiltersOptions>(filters)}
        >
            <FilterDate
                title={i18n('Period')}
                type={RidesFilter.DATE}
                filters={filters}
                onChangeFilter={onChangeFilterHandler}
            />
        </FiltersContainer>
    );
};
