import * as React from 'react';
import cn from 'classnames';

import { fetchAllSignals } from 'widgets/SignalsContentHeader/api/fetchAllSignals/fetchAllSignals';

import { DownloadSection, DownloadXLS } from 'features/DownloadXLS';
import { downloadXLS } from 'features/DownloadXLS/helpers/downloadXLS/downloadXLS';
import { SaveReport } from 'features/SaveReport';

import { ReportSection } from 'entities/Report/consts/ReportSection';
import { buildSignalXLSX } from 'entities/Signal/helpers/buildSignalXLSX/buildSignalXLSX';
import { formatServerSignals } from 'entities/Signal/helpers/formatServerSignals/formatServerSignals';

import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';

import { SignalsFiltersOptions } from 'components/Signals';
import { SIGNALS_TABLE_REDESIGN_HEADER } from 'components/Signals/SignalsTable';

import styles from 'widgets/SignalsContentHeader/ui/SignalsContentHeader/SignalsContentHeader.css';

export interface SignalsContentHeaderProps {
    className?: string;
    filters: SignalsFiltersOptions;
}

export const SignalsContentHeader: React.FC<SignalsContentHeaderProps> = function SignalsContentHeader({
    className,
    filters,
}) {
    const hasFilters = hasActiveFilters<SignalsFiltersOptions>(filters);

    const filtersValue = React.useMemo(
        () =>
            ['signals', 'actual_only'].reduce((result, key) => {
                if (filters?.[key]) {
                    result[key] = filters?.[key];
                }

                return result;
            }, {} as SignalsFiltersOptions),
        [filters],
    );

    const onDownloadXLS = React.useCallback(async () => {
        const data = await fetchAllSignals(filters);
        const signals = formatServerSignals(data.signals, data.signals_descriptions);

        if (data.can_get_more_pages) {
            // @TODO когда дойдем до лимита, уже точно надо перенести на сервер создание xml
            console.error('XLS download limit reached 5000');
        }

        if (signals) {
            downloadXLS({
                table: buildSignalXLSX(signals),
                headers: SIGNALS_TABLE_REDESIGN_HEADER,
                name: `signals_${Date.now()}`,
            });
        }
    }, [filters]);

    return (
        <div className={cn(styles.signalsContentHeader, className)}>
            <DownloadXLS
                section={DownloadSection.SIGNALS}
                onClick={onDownloadXLS}
            />

            {hasFilters && (
                <SaveReport
                    section={ReportSection.SIGNALS}
                    filters={filtersValue}
                />
            )}
        </div>
    );
};
