import * as React from 'react';
import { useHistory, useLocation } from 'react-router-dom';

import { getSignalsSourceFilterFlag } from 'utils/getSignalsSourceFilterFlag';

import { ClearFilters } from 'features/ClearFilters';
import { FilterDate } from 'features/FilterDate/ui/FilterDate/FilterDate';

import { SignalsActualFilter, SignalsDescriptionsFilter } from 'entities/Signal';
import {
    BOOLEAN_SIGNALS_FILTERS,
    DATE_SIGNALS_FILTERS,
    EXISTING_SIGNALS_FILTERS,
    INITIAL_SIGNALS_FILTERS,
    STRING_SIGNALS_FILTERS,
} from 'entities/Signal/consts/filters';
import { SignalsFilter } from 'entities/Signal/consts/SignalsFilter';
import { SignalsCarFilter } from 'entities/Signal/ui/SignalsCarFilter/SignalsCarFilter';
import { SignalsLevelFilter } from 'entities/Signal/ui/SignalsLevelFilter/SignalsLevelFilter';
import { SignalsSourceFilter } from 'entities/Signal/ui/SignalsSourceFilter/SignalsSourceFilter';

import { DateFilter } from 'shared/consts/DateFilter';
import { hasActiveFilters } from 'shared/helpers/hasActiveFilters/hasActiveFilters';
import { updateFiltersParams } from 'shared/helpers/updateFiltersParams/updateFiltersParams';
import { useSearchFilters } from 'shared/hooks/useSearchFilters/useSearchFilters';
import { FiltersContainer, FiltersContainerProps } from 'shared/ui/FiltersContainer/FiltersContainer';

import { SignalsFiltersOptions } from 'components/Signals';

import { i18n } from 'widgets/SignalsFilters/ui/SignalsFilters/SignalsFilters.i18n';

interface SignalsFiltersProps extends Pick<FiltersContainerProps, 'offsetTop'> {
    className?: string;

    initialFilters?: SignalsFiltersOptions;
    onFiltersChange?(filters: SignalsFiltersOptions): void;
}

// @todo: add storybook
export const SignalsFilters: React.FC<SignalsFiltersProps> = function SignalsFilters({
    className,
    offsetTop,
    initialFilters = INITIAL_SIGNALS_FILTERS,
    onFiltersChange,
}) {
    const location = useLocation();
    const history = useHistory();

    const filters = useSearchFilters<SignalsFiltersOptions>(initialFilters, EXISTING_SIGNALS_FILTERS, {
        stringFilters: STRING_SIGNALS_FILTERS,
        booleanFilters: BOOLEAN_SIGNALS_FILTERS,
        dateFilters: DATE_SIGNALS_FILTERS,
    });

    React.useEffect(() => {
        if (onFiltersChange) {
            onFiltersChange(filters);
        }
    }, [filters]);

    const onChangeFilterHandler = React.useCallback(
        (type: string, value: Optional<string | boolean> | Record<DateFilter, Nullable<number>>) => {
            const searchParams = new URLSearchParams(location?.search);

            history.push(`${location.pathname}?${updateFiltersParams(searchParams, type, value)}`);
        },
        [location],
    );

    return (
        <FiltersContainer
            className={className}
            offsetTop={offsetTop}
            clearButton={<ClearFilters filters={EXISTING_SIGNALS_FILTERS} />}
            isClearVisible={hasActiveFilters<SignalsFiltersOptions>(filters)}
        >
            <SignalsCarFilter
                value={filters[SignalsFilter.CAR]}
                onChangeFilter={onChangeFilterHandler}
            />

            <SignalsActualFilter
                checked={Boolean(filters[SignalsFilter.ACTUAL_ONLY])}
                onChangeFilter={onChangeFilterHandler}
            />

            <SignalsLevelFilter
                values={filters[SignalsFilter.PRIORITY]}
                onChangeFilter={onChangeFilterHandler}
            />

            <FilterDate
                title={i18n('Period')}
                type={SignalsFilter.DATE}
                filters={filters}
                onChangeFilter={onChangeFilterHandler}
            />

            {Boolean(getSignalsSourceFilterFlag()) && (
                <SignalsSourceFilter
                    values={filters[SignalsFilter.SOURCE]}
                    onChangeFilter={onChangeFilterHandler}
                />
            )}

            <SignalsDescriptionsFilter
                values={filters[SignalsFilter.SIGNALS]}
                onChangeFilter={onChangeFilterHandler}
            />
        </FiltersContainer>
    );
};
