import * as React from 'react';
import { useState } from 'react';
import cn from 'classnames/bind';

import { SignalDetailsModal } from 'features/SignalDetails';
import { SignalsTable } from 'features/SignalsTable/ui/SignalsTable/SignalsTable';

import { fetchSignal } from 'entities/Signal/api/fetchSignal/fetchSignal';
import { SignalFormattedSchema } from 'entities/Signal/types/SignalFormattedSchema';

import { HttpStatusCode } from 'shared/consts/HttpStatusCode';
import { KEY_LEFT, KEY_RIGHT } from 'shared/consts/keys';
import { addEventListener } from 'shared/helpers/addEventListener/addEventListener';
import { FetchError } from 'shared/helpers/fetchRequest/fetchRequest';
import { getFetchErrorMessage } from 'shared/helpers/getFetchErrorMessage/getFetchErrorMessage';
import { useCacheRequestContext } from 'shared/hooks/useCacheRequestContext/useCacheRequestContext';
import { useSearchParam } from 'shared/hooks/useSearchParam/useSearchParam';
import { FormErrorSchema } from 'shared/types/FormErrorSchema';
import { Button } from 'shared/ui/Button/Button';
import { ErrorMessage } from 'shared/ui/ErrorMessage/ErrorMessage';
import { Spin } from 'shared/ui/Spin/Spin';

import { i18n } from 'widgets/SignalsTableWidget/ui/SignalsTableWidget/SignalsTableWidget.i18n';

import ArrowLong from 'shared/ui/Icons/images/arrow-long-forward-24.inline.svg';

import styles from 'widgets/SignalsTableWidget/ui/SignalsTableWidget/SignalsTableWidget.css';

export interface SignalsTableWidgetProps {
    className?: string;

    data?: SignalFormattedSchema[];
    isFullyLoaded?: boolean;
    isLoading?: boolean;
    fetchNextPage?: () => void;
    onRemoveData?: (tag_id: string) => void;
    onUpdateData?: (tag_id: string, newData: Partial<SignalFormattedSchema>) => void;

    isShortView?: boolean;
    isOneCar?: boolean;
}

const cx = cn.bind(styles);

export const SignalsTableWidget: React.FC<SignalsTableWidgetProps> = React.memo(function SignalsTableWidget({
    className,
    data,
    isFullyLoaded,
    isLoading,
    isShortView,
    isOneCar,
    fetchNextPage,
    onRemoveData,
    onUpdateData,
}) {
    const cache = useCacheRequestContext();
    const [currentSignal, setCurrentSignal] = useState<SignalFormattedSchema>();

    const [signalId, setSignalId] = useSearchParam('signal_id');
    const [signalError, setSignalError] = useState<FormErrorSchema>();

    React.useEffect(() => {
        if (!signalId) {
            setCurrentSignal(undefined);
            setSignalError(undefined);

            return;
        }

        if (!currentSignal || currentSignal.tag_id !== signalId) {
            fetchSignal(signalId, cache)
                .ready()
                .then((data) => {
                    setCurrentSignal(data);
                    setSignalError(undefined);
                })
                .catch((error) => {
                    setCurrentSignal(undefined);

                    if (error instanceof FetchError && error.status === HttpStatusCode.NOT_FOUND) {
                        setSignalError([{ code: 'not_found', message: i18n('Signal not found') }]);
                    } else {
                        setSignalError(getFetchErrorMessage(error));
                    }
                });
        }
    }, [signalId, cache]);

    const updateSignal = React.useCallback(
        (signal: Optional<SignalFormattedSchema>) => {
            setCurrentSignal(signal);
            setSignalId(signal?.tag_id);
            setSignalError(undefined);
        },
        [setSignalId],
    );

    const onModalClose = React.useCallback(() => {
        updateSignal(undefined);
    }, [updateSignal]);

    const setNextSignal = React.useCallback(
        (diff: number) => {
            if (!data || !currentSignal) {
                onModalClose();

                return;
            }

            let currentIndex = diff + data.findIndex((item) => item.tag_id === currentSignal.tag_id);

            if (currentIndex >= data.length) {
                currentIndex = 0;
            } else if (currentIndex < 0) {
                currentIndex = data.length - 1;
            }

            updateSignal(data[currentIndex]);
        },
        [onModalClose, updateSignal, currentSignal, data],
    );

    const onPrevClick = React.useCallback(() => {
        setNextSignal(-1);
    }, [setNextSignal]);

    const onNextClick = React.useCallback(() => {
        setNextSignal(1);
    }, [setNextSignal]);

    const onKeyDownHandler = React.useCallback(
        (event: KeyboardEvent) => {
            if (!currentSignal) {
                return;
            }

            const { keyCode } = event;

            if (keyCode === KEY_LEFT) {
                onPrevClick();
            }

            if (keyCode === KEY_RIGHT) {
                onNextClick();
            }
        },
        [currentSignal, onNextClick, onPrevClick],
    );

    React.useEffect(() => {
        return addEventListener(document, 'keydown', onKeyDownHandler);
    }, [onKeyDownHandler]);

    return (
        <>
            <SignalsTable
                data={data}
                isLoading={isLoading}
                isFullyLoaded={isFullyLoaded}
                fetchNextPage={fetchNextPage}
                onClick={updateSignal}
                isShortView={isShortView}
                isOneCar={isOneCar}
            />

            {currentSignal && (
                <SignalDetailsModal
                    key={currentSignal.tag_id}
                    signal={currentSignal}
                    onClose={onModalClose}
                    onNextClick={onNextClick}
                    onRemoveData={onRemoveData}
                    onUpdateData={onUpdateData}
                >
                    <Button
                        className={cx(styles.arrow, styles.arrowLeft)}
                        round
                        shadow
                        icon={ArrowLong}
                        onClick={onPrevClick}
                    />

                    <Button
                        className={cx(styles.arrow, styles.arrowRight)}
                        round
                        shadow
                        icon={ArrowLong}
                        onClick={onNextClick}
                    />
                </SignalDetailsModal>
            )}

            {Boolean(!currentSignal && signalId) && (
                <SignalDetailsModal onClose={onModalClose}>
                    {!signalError && <Spin className={styles.loading} />}
                    {signalError && <ErrorMessage error={signalError} />}
                </SignalDetailsModal>
            )}
        </>
    );
});
