import { writeFile } from 'fs/promises';
import { join } from 'path';
import { walkFiles } from './walkFiles';
import { parseCodeFile } from './parseCodeFile';
import * as t from '@babel/types';
import traverse from '@babel/traverse';
import { cjs } from './cjs';
import { i18n as i18nBuilder } from 'mova-i18n';

type TranslatePlural = { one: string; some: string; many?: string; none?: string };

type TranslatesCache = Record<string, { ru: Array<string | TranslatePlural>; en: Array<string | TranslatePlural> }>;
interface SourceCache {
    i18nNames: Record<string, boolean>;
    brokenFiles: Record<string, boolean>;
}

function ast2obj(node: t.ObjectExpression): Record<string, any> {
    return node.properties.reduce((memo, prop) => {
        if (t.isObjectProperty(prop) && (t.isIdentifier(prop.key) || t.isStringLiteral(prop.key))) {
            const key = t.isIdentifier(prop.key) ? prop.key.name : prop.key.value;

            if (t.isStringLiteral(prop.value)) {
                memo[key] = prop.value.value;
            } else if (t.isObjectExpression(prop.value)) {
                memo[key] = ast2obj(prop.value);
            }
        }

        return memo;
    }, {});
}

async function _processFileStore(filename: string, cache: TranslatesCache): Promise<void> {
    if (filename.endsWith('.i18n/en.ts') || filename.endsWith('.i18n/ru.ts')) {
        const ast = await parseCodeFile(filename);

        const lang = filename.endsWith('.i18n/en.ts') ? 'en' : 'ru';

        cjs(traverse)(ast, {
            VariableDeclarator(path) {
                const node = path.node;

                if (t.isObjectExpression(node.init)) {
                    const data: Record<string, string | TranslatePlural> = ast2obj(node.init);

                    for (const key of Object.keys(data)) {
                        const value = data[key];

                        if (!cache[key]) {
                            cache[key] = { ru: [], en: [] };
                        }

                        if (cache[key][lang] && !cache[key][lang].includes(value)) {
                            cache[key][lang].push(value);
                        }
                    }
                }
            },
        });
    }
}

const allowList = {
    useI18N: true,
    getI18nLocale: true,
    setI18nLang: true,
    i18nBuilder: true,
};

async function _processFileSource(filename: string, cache: SourceCache): Promise<void> {
    if (filename.endsWith('.ts') || filename.endsWith('.tsx')) {
        const ast = await parseCodeFile(filename);

        cjs(traverse)(ast, {
            CallExpression(path) {
                const node = path.node;

                if (
                    t.isIdentifier(node.callee) &&
                    (node.callee.name === 'i18n' ||
                        (node.callee.name.toLowerCase().includes('i18n') && !allowList[node.callee.name]))
                ) {
                    if (node.arguments.length && t.isStringLiteral(node.arguments[0])) {
                        cache.i18nNames[node.callee.name] = true;
                    } else {
                        if (!(t.isIdentifier(node.arguments[0]) && node.arguments[0].name === 'keyset')) {
                            // warn
                            cache.brokenFiles[filename] = true;
                        }
                    }
                }
            },
        });
    }
}

export async function analize(): Promise<void> {
    console.log('PATH:', join(process.cwd(), 'src'));
    console.log('--------------');

    const cacheStore: TranslatesCache = {};
    const cacheSource: SourceCache = { brokenFiles: {}, i18nNames: {} };

    // initial processing
    for await (const filename of walkFiles(join(process.cwd(), 'src'))) {
        await _processFileStore(filename, cacheStore);
        await _processFileSource(filename, cacheSource);
    }

    console.log('CACHE:', cacheStore);
    console.log('--------------');

    let errorCount = 0;
    for (const key of Object.keys(cacheStore)) {
        const firstValue: string | TranslatePlural = cacheStore[key].ru[0];

        if (
            (typeof firstValue === 'string' && firstValue !== key) ||
            (typeof firstValue === 'object' && firstValue.one !== key)
        ) {
            console.log(key, cacheStore[key]);
            errorCount++;
        }
    }
    console.log('--------------');

    for (const key of Object.keys(cacheStore)) {
        if (cacheStore[key].en.length !== 1 || cacheStore[key].ru.length !== 1) {
            console.log(key, cacheStore[key]);
            errorCount++;
        }
    }
    console.log('--------------');

    console.log('TOTAL:', Object.keys(cacheStore).length, 'ERRORS:', errorCount);

    console.log('--------------');

    console.log(cacheSource);

    await writeFile(process.cwd() + '/tools/i18n/i18n-db.json', JSON.stringify(cacheStore, null, 2), 'utf-8');
}
