import { readFile, stat, rm } from 'fs/promises';
import { join } from 'path';
import { walkFiles } from './walkFiles';
import { parseCodeFile } from './parseCodeFile';
import * as t from '@babel/types';
import traverse from '@babel/traverse';
import { cjs } from './cjs';
import { generateCode } from './generateCode';
import { writeCodeFile } from './writeCodeFile';
import { Options as PrettierConfig } from 'prettier';

type TranslatePlural = { one: string; some: string; many?: string; none?: string };

type TranslatesCache = Record<string, { ru: Array<string | TranslatePlural>; en: Array<string | TranslatePlural> }>;

function basePath(str: string): string {
    return str.split('/src/')[1] || '';
}

async function _processFileSource(
    filename: string,
    cache: TranslatesCache,
    prettierConfig: PrettierConfig,
): Promise<void> {
    if ((filename.endsWith('.ts') || filename.endsWith('.tsx')) && !filename.includes('.story.')) {
        const i18nStoreFile = filename.replace(/.tsx?$/, '.i18n');
        let isConverted = true;

        try {
            await stat(i18nStoreFile + '.ts');
        } catch (error) {
            isConverted = false;
        }

        if (!isConverted) {
            const ast = await parseCodeFile(filename);

            const translates = {};
            let i18nNames = new Set<string>();
            let changed = false;

            cjs(traverse)(ast, {
                ImportDeclaration(path) {
                    const node = path.node;

                    if (
                        node.source &&
                        (node.source.value === '@yandex-int/i18n' || node.source.value === '@yandex-int/i18n/lib/react')
                    ) {
                        // import i18n from '@yandex-int/i18n';
                        // import { useI18N } from '@yandex-int/i18n/lib/react';
                        node.source.value = basePath(i18nStoreFile);
                        node.specifiers = [t.importSpecifier(t.identifier('i18n'), t.identifier('i18n'))];
                    } else if (
                        // import * as keyset from 'features/CarBooking/ui/CarCancelBookingDialog/CarCancelBookingDialog.i18n';
                        node.source &&
                        node.source.value.endsWith('.i18n')
                    ) {
                        path.remove();
                        changed = true;
                    }
                },

                CallExpression(path) {
                    const node = path.node;

                    if (t.isIdentifier(node.callee)) {
                        // const i18n = useI18N(keyset);
                        // const i18nText = i18n(keyset);
                        if (
                            (node.callee.name === 'useI18N' || node.callee.name === 'i18n') &&
                            node.arguments.length === 1 &&
                            t.isIdentifier(node.arguments[0]) &&
                            node.arguments[0].name === 'keyset'
                        ) {
                            const parentNode = path.parent;

                            if (t.isVariableDeclarator(parentNode) && t.isIdentifier(parentNode.id)) {
                                if (!i18nNames.has(parentNode.id.name)) {
                                    i18nNames.add(parentNode.id.name);
                                }

                                if (t.isVariableDeclaration(path.parentPath.parent)) {
                                    path.parentPath.parentPath.remove();
                                    changed = true;
                                }
                            }
                        }

                        // i18nText('Выберите автомобиль')
                        // i18n('Выберите автомобиль')
                        // i18n('{value} км', { value: formatNumber(Number(mileage)) })
                        if (
                            i18nNames.has(node.callee.name) &&
                            node.arguments.length >= 1 &&
                            t.isStringLiteral(node.arguments[0])
                        ) {
                            node.callee.name = 'i18n';

                            const text = node.arguments[0].value;

                            if (cache.hasOwnProperty(text) && cache[text].en.length >= 1) {
                                const variant = cache[text].en[0];
                                const translateKey = typeof variant === 'string' ? variant : variant.one;

                                node.arguments[0].value = typeof variant === 'string' ? variant : variant.one;

                                translates[translateKey] = {
                                    en: cache[text].en[0],
                                    ru: cache[text].ru[0],
                                };

                                if (cache[text].en.length > 1) {
                                    console.log(
                                        `[!!!] [${basePath(filename)}] "${text}" => ${cache[text].en
                                            .map((v) => JSON.stringify(v))
                                            .join(', ')}`,
                                    );
                                }

                                changed = true;
                            } else {
                                console.log(`[!!!] [${basePath(filename)}] "${text}" => ???`);
                            }
                        }
                    }
                },
            });

            if (changed) {
                await Promise.all([
                    writeCodeFile(filename, generateCode(ast), prettierConfig),
                    writeCodeFile(
                        filename.replace(/\.tsx?$/, '.i18n.ts'),
                        defaultStoreTemplate(translates),
                        prettierConfig,
                    ),
                    // rm(filename.replace(/\.tsx?$/, '.i18n'), { recursive: true }),
                ]);
            }
        }
    }
}

function defaultStoreTemplate(nextTranslates): string {
    const translates = Object.entries(nextTranslates)
        .sort((a, b) => {
            return a[0].localeCompare(b[0]);
        })
        .reduce((memo, [key, value]) => {
            memo[key] = value;
            return memo;
        }, {});

    return `
import { i18n as i18nBuilder, keyset, plurals, MovaLang } from 'mova-i18n';

const translates = keyset(${JSON.stringify(translates, null, 4)});

export const i18n = i18nBuilder(process.env.LANG as MovaLang, plurals)(translates);
`.trim();
}

export async function codeshift(): Promise<void> {
    const prettier = await readFile(process.cwd() + '/.prettierrc.json', 'utf-8');
    const content = await readFile(process.cwd() + '/tools/i18n/i18n-db.json', 'utf-8');
    const prettierConfig: PrettierConfig = JSON.parse(prettier);
    const cacheStore: TranslatesCache = JSON.parse(content);

    // initial processing
    const srcList = ['src'];

    for (const src of srcList) {
        for await (const filename of walkFiles(join(process.cwd(), src))) {
            await _processFileSource(filename, cacheStore, prettierConfig);
        }
    }
}
