import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoUser } from 'amazon-cognito-identity-js';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface ConfirmRegistrationControllerReq {
    email: string;
    code: string;
}

export async function confirmRegistrationController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<ConfirmRegistrationControllerReq>(req);
    const email = data.email;
    const code = data.code;

    if (!email || !code) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
                code: !code ? 'The field must be filled out' : undefined,
            },
        });
    }

    const cognitoUser = new CognitoUser({
        Username: email,
        Pool: userPull,
    });

    return new Promise((resolve) => {
        cognitoUser.confirmRegistration(code, true, function (error) {
            if (error) {
                sendResJson(res, HttpStatusCode.BAD_REQUEST, {
                    code: HttpStatusCode.BAD_REQUEST,
                    error: {
                        code: error.message,
                    },
                });

                return resolve();
            }

            sendResJson(res, HttpStatusCode.OK, {
                code: HttpStatusCode.OK,
            });

            return resolve();
        });
    });
}
