import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoUser } from 'amazon-cognito-identity-js';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface ConfirmRestoreControllerReq {
    email: string;
    code: string;
    newPassword: string;
}

export async function confirmRestoreController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<ConfirmRestoreControllerReq>(req);
    const email = data.email;
    const code = data.code;
    const newPassword = data.newPassword;

    if (!email || !code || !newPassword) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
                code: !code ? 'The field must be filled out' : undefined,
                newPassword: !newPassword ? 'The field must be filled out' : undefined,
            },
        });
    }

    const cognitoUser = new CognitoUser({
        Username: email,
        Pool: userPull,
    });

    return new Promise((resolve) => {
        cognitoUser.confirmPassword(code, newPassword, {
            onSuccess(data) {
                sendResJson(res, HttpStatusCode.OK, {
                    code: HttpStatusCode.OK,
                    data,
                });

                resolve();
            },
            onFailure(error) {
                sendResJson(res, HttpStatusCode.BAD_REQUEST, {
                    code: HttpStatusCode.BAD_REQUEST,
                    error: {
                        newPassword: error.message,
                    },
                });

                resolve();
            },
        });
    });
}
