import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoRefreshToken, CognitoUser, CognitoUserSession } from 'amazon-cognito-identity-js';
import { tokenVerifier, userPull } from 'controllers/helpers/cognito';
import { processUserSessionCookies } from 'controllers/helpers/processUserSessionCookies';
import cookie from 'cookie';
import { CurrentUserSchema, RequestHandlerOptions } from 'types';

import { ID_TOKEN_COOKIE_NAME, REFRESH_TOKEN_COOKIE_NAME } from 'consts/consts';

import { getReqHeaderStr } from 'lib/request/getReqHeaderStr/getReqHeaderStr';

export async function processIdToken(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<CurrentUserSchema | undefined> {
    const cookies = cookie.parse(getReqHeaderStr(req, 'cookie') || '');
    const token = cookies[ID_TOKEN_COOKIE_NAME];
    const rtoken = cookies[REFRESH_TOKEN_COOKIE_NAME];

    if (token) {
        try {
            return tokenVerifier.verify(token) as unknown as CurrentUserSchema;
        } catch (error) {
            // noop
            // try to restore user identity from refresh token
        }
    }

    if (rtoken) {
        const refreshToken = new CognitoRefreshToken({
            RefreshToken: rtoken,
        });

        const cognitoUser = new CognitoUser({
            Username: '',
            Pool: userPull,
        });

        return new Promise<CurrentUserSchema | undefined>((resolve) => {
            cognitoUser.refreshSession(refreshToken, (error, session: CognitoUserSession) => {
                if (error) {
                    return resolve(undefined);
                }

                processUserSessionCookies(session, res, { logger });
                resolve(session.getIdToken().decodePayload() as CurrentUserSchema);
            });
        });
    }

    return undefined;
}
