import { IncomingMessage, ServerResponse } from 'node:http';

import cookie from 'cookie';
import { RequestHandlerOptions } from 'types';

import { DM_DOMAIN, DM_SECURE_COOKIE, VUID_COOKIE_NAME } from 'consts/consts';

import { getReqHeaderStr } from 'lib/request/getReqHeaderStr/getReqHeaderStr';
import { generateVuid } from 'lib/vuid/generateVuid/generateVuid';
import { validateVuid } from 'lib/vuid/validateVuid/validateVuid';

// eslint-disable-next-line @typescript-eslint/no-magic-numbers
const VUID_MAX_AGE = 10 * 365 * 24 * 60 * 60; // 10 years

export async function processVuid(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<string> {
    const cookies = cookie.parse(getReqHeaderStr(req, 'cookie') || '');
    let vuid = cookies[VUID_COOKIE_NAME];

    if (!vuid || !validateVuid(vuid)) {
        vuid = await generateVuid();

        res.setHeader(
            'set-cookie',
            cookie.serialize(VUID_COOKIE_NAME, vuid, {
                domain: DM_DOMAIN,
                path: '/',
                maxAge: VUID_MAX_AGE,
                httpOnly: true,
                secure: DM_SECURE_COOKIE,
            }),
        );
    }

    return vuid;
}
