import { IncomingMessage, ServerResponse } from 'node:http';

import { AuthenticationDetails, CognitoUser } from 'amazon-cognito-identity-js';
import { authCallbacks } from 'controllers/helpers/authCallbacks';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface LoginControllerReq {
    email: string;
    password: string;
}

export async function loginController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<LoginControllerReq>(req);

    const email = data.email;
    const password = data.password;

    if (!email || !password) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
                password: !password ? 'The field must be filled out' : undefined,
            },
        });
    }

    const authenticationDetails = new AuthenticationDetails({
        Username: email,
        Password: password,
    });

    const cognitoUser = new CognitoUser({
        Username: email,
        Pool: userPull,
    });

    return new Promise((resolve) => {
        cognitoUser.authenticateUser(authenticationDetails, authCallbacks(req, res, resolve, { logger }));
    });
}
