import { IncomingMessage, ServerResponse } from 'node:http';

import { AuthenticationDetails, CognitoUser } from 'amazon-cognito-identity-js';
import { authCallbacks } from 'controllers/helpers/authCallbacks';
import { userPull } from 'controllers/helpers/cognito';
import { RequestHandlerOptions } from 'types';

import { HttpStatusCode } from 'consts/HttpStatusCode';

import { parseReqStream } from 'lib/request/parseReqStream/parseReqStream';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

interface PasswordResetControllerReq {
    email: string;
    password: string;
    newPassword: string;
}

export async function passwordResetController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const data = await parseReqStream<PasswordResetControllerReq>(req);
    const email = data.email;
    const password = data.password;
    const newPassword = data.newPassword;

    if (!email || !password || !newPassword) {
        return sendResJson(res, HttpStatusCode.BAD_REQUEST, {
            code: HttpStatusCode.BAD_REQUEST,
            error: {
                email: !email ? 'The field must be filled out' : undefined,
                password: !password ? 'The field must be filled out' : undefined,
                newPassword: !newPassword ? 'The field must be filled out' : undefined,
            },
        });
    }

    const authenticationDetails = new AuthenticationDetails({
        Username: email,
        Password: password,
    });

    const cognitoUser = new CognitoUser({
        Username: email,
        Pool: userPull,
    });

    return new Promise((resolve) => {
        const callbacks = authCallbacks(req, res, resolve, { logger });

        cognitoUser.authenticateUser(authenticationDetails, {
            ...callbacks,
            newPasswordRequired() {
                cognitoUser.completeNewPasswordChallenge(password, {}, callbacks);
            },
        });
    });
}
