import { IncomingMessage, ServerResponse } from 'node:http';

import { CognitoRefreshToken, CognitoUser, CognitoUserSession } from 'amazon-cognito-identity-js';
import { userPull } from 'controllers/helpers/cognito';
import { processUserSession } from 'controllers/helpers/processUserSession';
import cookie from 'cookie';
import { RequestHandlerOptions } from 'types';

import { REFRESH_TOKEN_COOKIE_NAME } from 'consts/consts';
import { HttpStatusCode } from 'consts/HttpStatusCode';

import { getReqHeaderStr } from 'lib/request/getReqHeaderStr/getReqHeaderStr';
import { sendResJson } from 'lib/response/sendResJson/sendResJson';

export async function refreshController(
    req: IncomingMessage,
    res: ServerResponse,
    { logger }: RequestHandlerOptions,
): Promise<void> {
    const cookies = cookie.parse(getReqHeaderStr(req, 'cookie') || '');
    const token = cookies[REFRESH_TOKEN_COOKIE_NAME];

    if (!token) {
        return sendResJson(res, HttpStatusCode.UNAUTHORIZED, {
            code: HttpStatusCode.UNAUTHORIZED,
            error: {
                _: 'Invalid refresh token',
            },
        });
    }

    const refreshToken = new CognitoRefreshToken({
        RefreshToken: token,
    });

    const cognitoUser = new CognitoUser({
        Username: '',
        Pool: userPull,
    });

    return new Promise((resolve) => {
        cognitoUser.refreshSession(refreshToken, (error, session: CognitoUserSession) => {
            if (error) {
                sendResJson(res, HttpStatusCode.UNAUTHORIZED, {
                    code: HttpStatusCode.UNAUTHORIZED,
                    error: {
                        _: 'Invalid refresh token',
                    },
                });

                return resolve();
            }

            processUserSession(session, req, res, { logger }).then(resolve, resolve);
        });
    });
}
